define('jira-agile/rapid/ui/chart/v2/releaseburndown/release-burndown-chart-data-service', ['require'], function(require) {
    'use strict';

    // REQUIRES
    var _ = require('underscore');
    var GlobalEvents = require('jira-agile/rapid/global-events');

    var Ajax;
    var RapidViewConfig;
    var ScopeBurndownBySprintTransformer;

    GlobalEvents.on('pre-initialization', function() {
        Ajax = GH.Ajax;
        RapidViewConfig = GH.RapidViewConfig;
        ScopeBurndownBySprintTransformer = GH.Reports.ScopeBurndownBySprintTransformer;
    });

    /**
     * Sends request to retrieve data
     *
     * @param {Object} requestParams
     * @returns {Promise}
     */
    function fetchData(requestParams) {
        var opts = {
            url: '/rapid/charts/releaseburndownchart',
            data: requestParams
        };
        return Ajax.get(opts, 'rapidChartData');
    }

    /**
     * Transform data to a form that the chart can render.
     *
     * @param {Object} data
     * @param {boolean} [zeroBaseline]
     * @returns {Object} transformed
     */
    function transformData(data, zeroBaseline) {
        return ScopeBurndownBySprintTransformer.getScopeBySprintData(data, zeroBaseline);
    }

    function ReleaseBurndownChartDataService() {}

    ReleaseBurndownChartDataService.prototype = {
        /**
         *
         * @param {number} rapidViewId
         * @param {number} versionId
         * @param {Object} filterOptions
         * @returns {Promise}
         */
        get: function(rapidViewId, versionId, filterOptions) {
            return this._getData(rapidViewId, versionId, filterOptions).andThen(function(data) {
                data.rapidViewId = RapidViewConfig.currentData.id;
                data.estimationStatistic = RapidViewConfig.currentData.data.estimationStatistic;
                data.estimationStatistic.formatter = ScopeBurndownBySprintTransformer.getEstimationStatisticFormatter(data.estimationStatistic.renderer);
                data.labels = {
                    originalEstimate: AJS.I18n.getText('gh.rapid.chart.versionreport.originalestimate'),
                    workAdded: AJS.I18n.getText('gh.rapid.work.added.version'),
                    workRemoved: AJS.I18n.getText('gh.rapid.work.removed.version'),
                    forecastDescription: function(sprintsRemaining) {
                        return AJS.I18n.getText('gh.rapid.charts.releaseburndown.forecastdescription', sprintsRemaining);
                    },
                    helpUrl: GH.HelpPaths.getHelpPath("viewing.release.burndown").url
                };

                return transformData(data, filterOptions.zeroBaseline);
            });
        },

        /**
         * Clear cached data
         */
        clearCache: function() {
            if (this._dataPromise) {
                this._dataPromise = null;
                this._dataRequestParams = null;
            }
        },

        /**
         * For testing
         *
         * @private
         */
        _transformData: transformData,

        /**
         * Gets raw data for given rapidViewId, versionId and filterOptions. Data could be retrieved from cache if available.
         *
         * @param {number} rapidViewId
         * @param {number} versionId
         * @param {object} [filterOptions]
         * @returns {Promise}
         * @private
         */
        _getData: function(rapidViewId, versionId, filterOptions) {
            var requestParams = {
                rapidViewId: rapidViewId,
                versionId: versionId
            };
            // Only send a request if previous one cannot be used
            // TODO need a way to purge the cache when navigating away from the report
            if (!this._dataPromise || this._dataPromise.state() === 'rejected' || !_.isEqual(this._dataRequestParams, requestParams)) {
                this._dataRequestParams = requestParams;
                this._dataPromise = fetchData(requestParams);
            }
            return this._dataPromise;
        }
    };

    return ReleaseBurndownChartDataService;
});
