/* global GH */
/**
 * Estimate Picker
 * @module jira-agile/rapid/ui/component/estimate-picker
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/ui/chart/chart-view
 * @requires module:jira/ajs/select/single-select
 */
define('jira-agile/rapid/ui/component/estimate-picker', ['require'], function(require) {

    // REQUIRES
    var $ = require('jquery');
    var _ = require('underscore');
    var ChartView = require('jira-agile/rapid/ui/chart/chart-view');
    var SingleSelect = require('jira/ajs/select/single-select');

    // GLOBALS... FIX ME
    var Ajax = GH.Ajax;
    var BoardState = GH.BoardState;
    var RapidBoardState = GH.RapidBoard.State;
    var tpl = GH.tpl;

    /**
     * Estimate picker dropdown implementation
     */
    var EstimatePicker = function (selectedItemKey, supportTimeTracking) {
        this.selectedItemStateKey = selectedItemKey;
        this.items = [];
        this.defaultItem = undefined;
        this.supportTimeTracking = supportTimeTracking;
    };


    /**
     * Set the items
     */
    EstimatePicker.prototype.setItems = function (items) {
        this.items = items || [];
    };

    /**
     * Get the items
     */
    EstimatePicker.prototype.getItems = function () {
        return this.items;
    };

    /**
     * Are there any items
     */
    EstimatePicker.prototype.hasItems = function () {
        return !_.isEmpty(this.items);
    };
    EstimatePicker.prototype.getDefaultItem = function () {
        return this.defaultItem;
    };
    EstimatePicker.prototype.getDefaultItemId = function () {
        return this.defaultItem ? this.defaultItem.id : false;
    };
    /**
     * Set the default item to load when no item has been selected
     * @param defaultItem an object representing one of the items in the list
     */
    EstimatePicker.prototype.setDefaultItem = function (defaultItem) {
        this.defaultItem = defaultItem || undefined;
    };
    /**
     * Get an estimate given its id (string)
     */
    EstimatePicker.prototype.getItemById = function (itemId) {
        var item = _.find(this.items, function (item) {
            return item.id === itemId;
        });
        return item;
    };
    /**
     * Get the currently selected item. This will automatically select the first active item if current selection
     * is invalid.
     * @return a object representing the item or false if no items are available
     */
    EstimatePicker.prototype.getSelectedItem = function () {
        var storedItemId = this.getStoredSelectedItemId();
        var item = storedItemId ? this.getItemById(storedItemId) : null;

        if (!item) {
            // select the default item if we have one
            if (this.getDefaultItem()) {
                item = this.getDefaultItem();
            } else if (this.hasItems()) {
                // fall back to last item
                item = _.last(this.getItems());
            }
            this.storeSelectedItemId();
            RapidBoardState.pushState();
        }

        return item || false;
    };

    /**
     * Set the new selected item
     */
    EstimatePicker.prototype.setSelectedItem = function (selectedItemId, preventEventTrigger) {
        this.previousSelectedItem = this.getSelectedItem();
        // fetch the item for the given id
        var item = this.getItemById(selectedItemId);
        if (item) {
            this.storeSelectedItemId(selectedItemId);
        } else {
            this.storeSelectedItemId();
        }
        if (!preventEventTrigger) {
            $(this).trigger('selectedItemChanged', {item: item});
        }
    };


    EstimatePicker.prototype.getStoredSelectedItemId = function () {
        // we keep the selected issues in the session selectedItemStateKey, avoids duplicating data in several places
        return BoardState.getPerViewValue(this.selectedItemStateKey, null);
    };

    EstimatePicker.prototype.storeSelectedItemId = function (selectedItemId) {
        BoardState.setPerViewValue(this.selectedItemStateKey, selectedItemId);
    };

//
// Rendering logic
//
    EstimatePicker.prototype.render = function (container) {
        // render the dropdown choices
        var hasDropdown = this.items.length > 1;
        var selectedItem = this.getSelectedItem();

        // after the frother control is disabled, we want to redraw this
        var self = this;
        var onSelect = function (event, selected) {
            self.setSelectedItem(selected.value());
        };

        var showDropdownAndRedraw = function (e) {
            self.showDropdown(e, function () {
                self.render(container);
            }, onSelect);
        };

        if (hasDropdown) {
            container.html(tpl.component.estimatePicker.renderSingleSelect({
                items: this.items,
                selectedItem: selectedItem
            }));
            $('#ghx-estimate-picker-trigger').click(showDropdownAndRedraw);
        } else if (selectedItem) {
            container.html(tpl.component.estimatePicker.renderSingleItem({
                selectedItemId: selectedItem.id,
                selectedItemName: selectedItem.name || ''
            }));
        }

        $('#ghx-estimate-picker-form').submit(function () {
            return false;
        });
    };

    EstimatePicker.prototype.showDropdown = function (e, redrawFn, selectFn) {
        $(e.target).hide();
        var $select = $('#ghx-estimate-picker');
        var singleSelect = new SingleSelect({
            element: $select,
            width: 250,
            errorMessage: '',
            setMaxHeightToWindow: true
        });

        var $field = $('#ghx-estimate-picker-field');

        $select.bind('selected', function (e, selected) {
            if (selected.value()) {
                selectFn(e, selected);
            }
            $field.blur();
        });

        $field.blur(redrawFn);

        $field.focus().select();

        // hack :( but I cannot find another way to force the frother control to show the suggestions
        // this seems to be the shortest timeout that works reliably
        setTimeout(function () {
            $('#ghx-estimate-picker-field').trigger('click');
        }, 20);

    };

    EstimatePicker.prototype.load = function (rapidViewId, onItemChange) {
        var picker = this;
        return Ajax.get({
            url: '/rapidviewconfig/estimation.json',
            data: {
                rapidViewId: rapidViewId
            }
        }, 'rapidBoardEstimateStatistics')
            .done(function (data) {
                if (data.availableEstimationStatistics) {
                    var statisticOptions = data.availableEstimationStatistics;
                    var defaultItem = data.currentEstimationStatistic;
                    if (picker.supportTimeTracking && data.currentTrackingStatistic.isEnabled) {
                        statisticOptions.push(data.currentTrackingStatistic);
                        defaultItem = data.currentTrackingStatistic;
                    }

                    // put them in the picker
                    picker.setItems(statisticOptions);

                    picker.setDefaultItem(defaultItem);

                    var itemChangeCallbackWithArgument = function () {
                        var selectedItem = picker.getSelectedItem().id;
                        onItemChange(selectedItem);
                    };
                    $(picker).bind('selectedItemChanged', itemChangeCallbackWithArgument);

                    picker.render(ChartView.getChartControls(true));
                    ChartView.showChartControls();
                }
                // else do nothing.  just use the configured estimate.
            });
    };

    return EstimatePicker;
});
