/**
 * Manages the page presentation mode, that is whether its the normal display (including page header),
 * compact display (page header hidden) or high contrast mode
 */

GH.PresentationMode = {};

GH.PresentationMode.NORMAL_MODE = 'normal';
GH.PresentationMode.COMPACT_MODE = 'compact';
GH.PresentationMode.PROJECTOR_MODE = 'projector';

/**
 * Available modes
 */
GH.PresentationMode.allModes = [
    GH.PresentationMode.NORMAL_MODE,
    GH.PresentationMode.COMPACT_MODE,
    GH.PresentationMode.PROJECTOR_MODE
];

/**
 * Currently selected mode
 */
GH.PresentationMode.mode = GH.PresentationMode.NORMAL_MODE;

GH.PresentationMode.enabled = false;

/**
 * Initialises this components
 */
GH.PresentationMode.init = function() {
    // initialise the stored presentation mode
    var storedMode = GH.PresentationMode.getStoredMode();
    this.mode = GH.PresentationMode.validateMode(storedMode);
    this.enabled = true;

    // update UI
    this._updateBodyClasses();
};

/**
 * Is compact mode currently enabled
 */
GH.PresentationMode.isHeaderHidden = function () {
    return this.mode === GH.PresentationMode.COMPACT_MODE || this.mode === GH.PresentationMode.PROJECTOR_MODE;
};

GH.PresentationMode.isNormalMode = function () {
    return this.mode === GH.PresentationMode.NORMAL_MODE;
};

GH.PresentationMode.isProjectorMode = function () {
    return this.mode === GH.PresentationMode.PROJECTOR_MODE;
};

GH.PresentationMode.toggleNormalAndCompact = function() {
    GH.PresentationMode.iterateThroughPresentationModes([
        GH.PresentationMode.NORMAL_MODE,
        GH.PresentationMode.COMPACT_MODE
    ]);
};

/**
 * Validates the passed mode and falls back to normal mode if the mode is unknown
 */
GH.PresentationMode.validateMode = function(mode) {
    if (!_.contains(GH.PresentationMode.allModes, mode)) {
        return GH.PresentationMode.NORMAL_MODE;
    }
    return mode;
};

/**
 * Sets the presentation mode to use
 */
GH.PresentationMode.setPresentationMode = function(mode) {
    if (!GH.PresentationMode.enabled) {
        return;
    }

    mode = GH.PresentationMode.validateMode(mode);

    // store mode and update ui
    this.mode = mode;
    GH.PresentationMode.setStoredMode(mode);
    this._updateUI();
};

/**
 * Iterates through a set of options, uses the first one if the current option isn't in the list
 * @param modes
 */
GH.PresentationMode.iterateThroughPresentationModes = function (modes) {
    var pos = _.indexOf(modes, this.mode);
    pos++;
    pos = pos % modes.length;
    GH.PresentationMode.setPresentationMode(modes[pos]);
};

/**
* Button to change presentation mode
*/
GH.PresentationMode.render = function(expand, containerSelector) {

    // render the button
    AJS.$(containerSelector).append(GH.tpl.presentationmode.renderPresentationButton({'expand' : expand}));

    // register click handler
    GH.PresentationMode._registerClickHandler();

    // tooltip
    GH.PresentationMode.tipsify();

};

/**
 * For some reason the click handler is "lost" when we use browser forward/backward buttons.
 * Just reattach handler every time the element is fetched
 */
GH.PresentationMode._registerClickHandler = function() {

    var self = this;
    AJS.$('.js-compact-toggle').unbind('click.presentationMode').bind('click.presentationMode', function() {
        self.toggleNormalAndCompact();
    });

};

GH.PresentationMode._updateUI = function() {
    // make sure we close any opened dropdowns
    AJS.$('body').click();

    // update classes on body, etc
    this._updateBodyClasses();

    // update sizing of all the things
    GH.RapidBoard.ViewController.handleResizeEvent();
};

GH.PresentationMode._updateBodyClasses = function() {
    // if already active close it else open it
    var $body = AJS.$('body');
    if(this.isHeaderHidden()) {
        $body.addClass('ghx-header-compact');
    } else {
        $body.removeClass('ghx-header-compact');
    }
    if (this.isProjectorMode()) {
        $body.addClass('ghx-contrast-high');
    } else {
        $body.removeClass('ghx-contrast-high');
    }
};

/**
 * Get the presentation mode stored in the session
 */
GH.PresentationMode.getStoredMode = function() {
    return GH.storage.get('gh.presentation.mode', true);
};

/**
 * Sets the stored presentation mode
 */
GH.PresentationMode.setStoredMode = function(mode) {
    GH.storage.put('gh.presentation.mode', mode, true);
};

/**
 * Tooltips
 */
GH.PresentationMode.tipsify = function() {

    GH.Tooltip.tipsify({
        selector: '.js-compact-toggle',
        gravity: 'e'
    });
};
