/* global _ */
/**
 * Controller for interaction with the LinkedPagesInlineDialog for "entities" (ex, epic, version, sprint)
 * @type {{}}
 */
GH.LinkedPagesController = {};

GH.LinkedPagesController.EPIC = "epic";
GH.LinkedPagesController.VERSION = "version";
GH.LinkedPagesController.SPRINT = "sprint";

GH.LinkedPagesController.PLAN = "plan";
GH.LinkedPagesController.REPORT = "report";

GH.LinkedPagesController.CONFLUENCE_CREATE_PAGE_ACTION = "/plugins/createcontent/init-dialog.action";

/**
 * The last created dialog. This dialog can be hidden so it is safer to check visibility of the dialog before assuming
 * it is shown.
 *
 * @type {GH.LinkedPagesInlineDialog}
 */
GH.LinkedPagesController._dialog = null;
GH.LinkedPagesController._authenticated = false;

(function () {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    GH.LinkedPagesController.analytics = {
        show: new AnalyticsTracker("gh.linkedPages.dialog.show"),
        count: new AnalyticsTracker("gh.linkedPages.dialog.count"),
        action: new AnalyticsTracker("gh.linkedPages.dialog.action")
    };
})();

/**
 * Tells if the active remote link creation conversations poller is running.
 *
 * @type {boolean}
 */

GH.LinkedPagesController._pollingConversations = false;

/**
 * Map of active remote link creation conversation tokens associated to their initialising dialog
 *
 * @type {Object.<string, GH.LinkedPagesInlineDialog>}
 */
GH.LinkedPagesController._dialogByLinkCreationToken = {};

/**
 * Binds necessary event handlers to a linked pages dialog trigger so that it will open
 * a dialog for the correct entity
 *
 * @param entityType
 * @param agileMode
 * @param entityId
 * @param canEdit
 * @param $trigger
 */
GH.LinkedPagesController.initDialogTrigger = function (entityType, agileMode, entityId, $trigger) {
    $trigger.on('click', GH.Util.wrapPreventDefault(function () {
        GH.LinkedPagesController._showDialog(entityType, agileMode, entityId, $trigger);
    }));
};

/**
 * Generic showDialog.
 *
 * @param entityType
 * @param agileMode
 * @param entityId
 * @param canEdit
 * @param $trigger
 * @private
 */
GH.LinkedPagesController._showDialog = function (entityType, agileMode, entityId, $trigger) {
    var linkedPagesDialog = new GH.LinkedPagesInlineDialog({
        entityType: entityType,
        agileMode: agileMode,
        entityId: entityId,
        trigger: $trigger
    });

    linkedPagesDialog.show();

    GH.LinkedPagesController.analytics.show.trigger([agileMode, entityType].join('.'));

    GH.LinkedPagesController._dialog = linkedPagesDialog;

    if (!GH.LinkedPagesController._authenticated) {
        AJS.$(document).bind('applinks.auth.completion', GH.LinkedPagesController._handleAuthenticationCompleted);
        GH.LinkedPagesController._authenticated = true;
    }
};

/**
 * Shows the linked pages inline dialog for an epic with the given issue key at the location of the given trigger jQuery object
 * TODO - this doesn't seem to be used at the moment? Are we going to migrate to it at some point?
 *
 * @param epicKey
 * @param canEditEpics
 * @param $trigger
 */
GH.LinkedPagesController.showDialogForEpic = _.partial(GH.LinkedPagesController._showDialog, GH.LinkedPagesController.EPIC);

/**
 * Shows the linked pages inline dialog for a sprint with the given sprint id at the location of the given trigger jQuery object
 *
 * @param sprintId
 * @param canEdit
 * @param $trigger
 */
GH.LinkedPagesController.showDialogForSprint = _.partial(GH.LinkedPagesController._showDialog, GH.LinkedPagesController.SPRINT);

/**
 * Hides the currently showing dialog, if any
 */
GH.LinkedPagesController.hideCurrentDialog = function () {
    GH.LinkedPagesController._dialog && GH.LinkedPagesController._dialog.hide();
};

/**
 * Fetches the linked pages for the entity of given type entityType and with entityId
 *
 * @param entityType
 * @param entityId
 * @returns {Deferred} A deferred object
 */
GH.LinkedPagesController.getLinkedPagesForEntity = function (entityType, entityId) {
    switch (entityType) {
        case GH.LinkedPagesController.EPIC:
            return GH.LinkedPagesController._getLinkedPagesForEpic(entityId);
        case GH.LinkedPagesController.VERSION:
            // TODO implement
            break;
        case GH.LinkedPagesController.SPRINT:
            return GH.LinkedPagesController._getLinkedPagesForSprint(entityId);
    }
};

GH.LinkedPagesController._getLinkedPagesForEpic = function (epicKey) {
    return GH.Ajax.get({
        url: "/epics/" + epicKey + "/pages"
    }, "getEpicLinkedPages").done(function (result) {
        GH.PageIssueLinksModel.set(epicKey, result.pages);
        GH.LinkedPagesController.analytics.count.trigger('epic', 'pages', result.pages.length);
    });
};

GH.LinkedPagesController._getLinkedPagesForSprint = function (sprintId) {
    return GH.Ajax.get({
        url: "/sprint/" + sprintId + "/pages"
    }, "getSprintLinkedPages").done(function (result) {
        GH.PageIssueLinksModel.set(sprintId, result.pages);
        GH.LinkedPagesController.analytics.count.trigger('sprint', 'pages', result.pages.length);
    });
};

/**
 * Starts a new conversation in the backend and enlist the conversation and the associated dialog to be watched for
 * completion by the poller.
 *
 * @param {GH.LinkedPagesInlineDialog} dialog the dilog that originated this conversation
 * @returns {AJS.$.Promise} resolved with the token of the newly created conversation
 * @see GH.LinkedPagesController._ensurePolling
 */
GH.LinkedPagesController.startRemoteLinkCreationConversation = function (dialog) {

    // we want to make sure the conversations completion is tracked
    GH.LinkedPagesController._ensurePolling();

    return GH.LinkedPagesService.initiateRemoteLinkConversation().done(function (token) {
        // enlist conversation to be watched
        GH.LinkedPagesController._dialogByLinkCreationToken[token] = dialog;
    });
};

/**
 * Removes a linked page from an entity.
 * An api call is made to execute the removal and if it's successfull, the linked pages model for the entity will be updated as well.
 *
 * @param entityType
 * @param entityId
 * @param linkId
 * @returns {Deferred} a deferred object
 */
GH.LinkedPagesController.removeLinkedPage = function (entityType, entityId, linkId) {
    var apiDeferred;
    switch (entityType) {
        case GH.LinkedPagesController.EPIC:
            apiDeferred = GH.Ajax.del({
                url: "/issuelink/" + linkId
            });
            break;
        case GH.LinkedPagesController.VERSION:
            break;
        case GH.LinkedPagesController.SPRINT:
            apiDeferred = GH.Ajax.del({
                url: "/api/sprints/" + entityId + "/remotelink/" + linkId
            });
            break;
    }
    apiDeferred.done(function () {
        // remove the page from the model
        GH.PageIssueLinksModel.removeLink(entityId, linkId);
        GH.LinkedPagesController.analytics.action.trigger([entityType, 'removeLink'].join('.'));
    });
    return apiDeferred;
};

/**
 * Performs a search for pages based on the given query.
 *
 * @param entityType
 * @param entityId
 * @param query
 * @returns {*}
 */
GH.LinkedPagesController.searchPages = function (entityType, entityId, query) {
    return GH.PageSearchRequestService.search(entityId, query).andThen(function (result) {
        // filter out the pages already linked to the entity
        var linkedPageIssueLinks = GH.PageIssueLinksModel.list(entityId);
        result.pages = _.reject(result.pages, function (page) {
            return _.findWhere(linkedPageIssueLinks, { pageId: page.id });
        });
        GH.PagesSearchModel.set(entityId, result.pages);
        return result;
    });
};

/**
 * Adds an issue link to a page with the given pageId to the entity with the given entityType and entityId.
 *
 * @param entityType
 * @param entityId
 * @param pageId
 * @returns {Promise} a promise
 */
GH.LinkedPagesController.addLink = function (entityType, entityId, pageId, pageTitle) {
    switch (entityType) {
        case GH.LinkedPagesController.EPIC:
            return GH.LinkedPagesController._addLinkForEpic(entityId, pageId);
        case GH.LinkedPagesController.VERSION:
            // TODO implement
            break;
        case GH.LinkedPagesController.SPRINT:
            return GH.LinkedPagesController._addLinkForSprint(entityId, pageId, pageTitle);
    }
};

GH.LinkedPagesController._addLinkForEpic = function (epicKey, pageId) {
    return GH.Ajax.post({
        url: "/epics/" + epicKey + "/pages",
        data: { pageId: pageId }
    }, "addConfluencePageToEpic").andThen(function (result) {
        var issueLink = result.success;
        GH.PageIssueLinksModel.addLink(epicKey, issueLink);
        GH.PagesSearchModel.removePage(epicKey, pageId);
        GH.LinkedPagesController.analytics.action.trigger('epic.addLink');
        return issueLink;
    });
};

GH.LinkedPagesController._addLinkForSprint = function (sprintId, pageId, pageTitle) {
    return GH.Ajax.post({
        url: "/sprint/" + sprintId + "/pages",
        data: { pageId: pageId, pageTitle: pageTitle }
    }, "addConfluencePageToSprint").andThen(function (result) {
        var issueLink = result.success;
        GH.PageIssueLinksModel.addLink(sprintId, issueLink);
        GH.PagesSearchModel.removePage(sprintId, pageId);
        GH.LinkedPagesController.analytics.action.trigger('sprint.addLink');
        return issueLink;
    });
};

GH.LinkedPagesController._handleAuthenticationCompleted = function (eventObject, applinkId, success, authAdminUri, wasRedispatched) {
    var dialog = GH.LinkedPagesController._dialog;
    if (success && dialog) {
        if (dialog.isSearchMode()) {
            dialog.showSearch(dialog.getSearchQuery());
            dialog.refresh();
        } else {
            dialog.show();
        }
    } else {
        GH.Notification.showWarning(AJS.I18n.getText('gh.entity.pages.server.authentication.failed'));
    }
};

/**
 * Ensure that the poller periodically checks enlisted conversation status. If not it starts a new one.
 */
GH.LinkedPagesController._ensurePolling = function () {

    if (!GH.LinkedPagesController._pollingConversations) {
        GH.Poller.addPoller('linkedPagePoller', function linkedPagePollerHandler() {
            GH.Ajax.get({
                url: '/remoteLinkConversation/statuses',
                traditional: true,
                data: { token: _.keys(GH.LinkedPagesController._dialogByLinkCreationToken) }
            }, 'remoteLinksConversationStatuses').done(function (tokenStatuses) {

                // get the redeemed tokens in this polling loop
                var redeemedTokens = _.pluck(_.where(tokenStatuses, { status: 'REDEEMED' }), 'token');

                // get a local copy of the map for tokens redeemed in this poller loop
                var dialogByLinkCreationToken = _.pick(GH.LinkedPagesController._dialogByLinkCreationToken, redeemedTokens);

                _.each(dialogByLinkCreationToken, function (dialog, token) {
                    // if the dialog for this token is the active we refresh it (show when visible acts like a refresh)
                    var activeDialog = GH.LinkedPagesController._dialog;
                    if (activeDialog === dialog && activeDialog.isVisible()) {
                        activeDialog.show();
                    }

                    // remove the mapping to the dialog from the map
                    delete GH.LinkedPagesController._dialogByLinkCreationToken[token];
                });
            });
        });

        GH.LinkedPagesController._pollingConversations = true;
    }
};