/**
 * Renders the various parts of the details view
 *
 * Manages the view/edit state of fields as well as general field rendering
 */
define('jira-agile/rapid/ui/detail/details-field-renderer', [
    'require'
], function(
    require
) {
    'use strict';

    var _ = require('underscore');
    var RapidViewConfig = GH.RapidViewConfig;
    var DetailsModel = GH.DetailsObjectFactory.getDetailsModel();

    var DetailsFieldRenderer = {};

    var PARENT_ISSUE_KEY_ID = 'parentissuekey';

    /**
     * Renders the fields on the details view.
     * @param viewData the view data of the current issue
     * @param detailContainer the div container for the detail view
     * @param detailViewOptions the options that the detail view was initialised with
     */
    DetailsFieldRenderer.renderFields = function (viewData, detailContainer, detailViewOptions) {

        // header section
        this.renderDetailsHeaderFields(viewData, detailContainer, detailViewOptions);

        // details tab
        var detailsSectionsMeta = [
            {fieldsSelector: 'details'},
            {fieldsSelector: 'people', title: AJS.I18n.getText('common.concepts.people')},
            {fieldsSelector: 'dates', title: AJS.I18n.getText('common.concepts.dates')}
        ];

        var detailsSections = _.reduce(detailsSectionsMeta, function (arr, sectionMeta) {
            var fields = DetailsModel.getFieldsForSection(sectionMeta.fieldsSelector);
            if (fields.length > 0) {
                arr.push({title: sectionMeta.title, fields: fields});
            }
            return arr;
        }, []);

        if (!GH.Features.EDITABLE_DETAIL_VIEW_ENABLED.isEnabled() && viewData.issueLinks.linkingEnabled) {
            detailsSections.push({
                title: AJS.I18n.getText('common.concepts.issuelinks'),
                issueLinks: viewData.issueLinks
            });
        }

        var detailTabs = detailContainer.find('#ghx-detail-nav');
        var detailView = detailContainer;

        // Tabs
        var tabs = this.buildTabs(viewData, detailsSections);

        // render the tabs and contained fields
        this.renderTabs(viewData, detailView, detailTabs, tabs);
    };

    /**
     * Renders the fields on the details view.
     * @param viewData the view data of the current issue
     * @param detailContainer the div container for the detail view
     * @param detailViewOptions the options that the detail view was initialised with
     */
    DetailsFieldRenderer.renderDetailsHeaderFields = function(viewData, detailContainer, detailViewOptions) {
        // initialise fields
        var issueKeyField = DetailsModel.getViewField('issuekey');
        var parentIssueKeyField = DetailsModel.getViewField(PARENT_ISSUE_KEY_ID);
        var summaryField = DetailsModel.getViewField('summary');

        // locate detail head container
        var detailHead = detailContainer.find('#ghx-detail-head');

        // Parent Issue Key
        this._renderIssueKey(detailHead, '.ghx-parent-key', parentIssueKeyField, viewData);

        // Issue Key
        this._renderIssueKey(detailHead, '.ghx-key', issueKeyField, viewData);

        // Summary
        var summaryDom = this.renderField(viewData, summaryField);
        detailHead.find('.ghx-detail-summary').empty().append(summaryDom);

        // Statistic
        this.renderEstimationAndTimeTracking(viewData, detailHead);

        // Log Work Trigger - only if we are in Work mode, have a Tracking Statistic, and the operation is allowed
        var showLogWorkTrigger = detailViewOptions.showLogWorkTrigger && DetailsModel.hasIssueOperation('log-work');
        this.renderLogWorkTrigger(viewData, detailHead, showLogWorkTrigger);
    };

    DetailsFieldRenderer._renderIssueKey = function (detailHead, selector, field, viewData) {
        if( !_.isUndefined(field) ) {
            var fieldToRender = _.extend({}, field, {
                dataTooltip: this._getIssueKeyTooltip(field)
            });

            var headerKey = detailHead.find(selector);
            var issueKeyDom = this.renderField(viewData, fieldToRender);
            headerKey.html(issueKeyDom);

            // render link to issue in JIRA
            this.renderJiraLink(viewData, fieldToRender, headerKey);
            this._initTooltip(issueKeyDom);
        }
    },

    DetailsFieldRenderer._initTooltip = function ($el) {
        $el.find('[data-tooltip]').each(function (idx, el) {
            AJS.$(el).tooltip({
                html: true,
                title: 'data-tooltip'
            })
        });
    },

    DetailsFieldRenderer._getIssueKeyTooltip = function (field) {
        if( field.id===PARENT_ISSUE_KEY_ID) {
            return GH.tpl.detailview.renderParentIssueTooltip({
                issueKey: field.text,
                summary: field.label
            });
        }
    },

    DetailsFieldRenderer.renderEstimationAndTimeTracking = function (viewData, detailHead) {
        GH.log('unimplemented function renderEstimationAndTimeTracking', GH.Logger.Contexts.ui);
    };

    /**
     * @param viewData the view data of the current issue
     * @return array of tab data to be rendered by renderTabs method
     */
    DetailsFieldRenderer.buildTabs = function(viewData) {
        GH.log('unimplemented function buildTabs', GH.Logger.Contexts.ui);
    };

    DetailsFieldRenderer.addSubtaskTab = function(tabs, viewData) {
        if (!viewData.subtasks) {
            return;
        }
        // we show the actions column if there is at least one action to show
        var showActionsColumn = !!_.find(viewData.subtasks, function (subtask) {
            return subtask.canEdit || subtask.canDelete;
        });

        // fetch the subtasks aggregate estimate
        var showTrackingField = !_.isUndefined(viewData.trackingStatisticFieldId);
        var trackingFieldSum;
        if (showTrackingField) {
            var subtaskAggregateEstimateField = DetailsModel.getViewField('aggregatetimeestimate_subtasks');
            if (subtaskAggregateEstimateField) {
                trackingFieldSum = subtaskAggregateEstimateField.html;
            }
        }

        tabs.push({
            id: 'subtasks',
            title: AJS.I18n.getText('gh.boards.subtasks'),
            titleCount: viewData.subtasks.length,
            hasTitleCount: true,
            subtasks: viewData.subtasks,
            canCreateSubtask: DetailsModel.hasIssueOperation('create-subtask'),
            showActionsColumn: showActionsColumn,
            showTrackingField: showTrackingField,
            trackingFieldSum: trackingFieldSum,
            filteredSubtaskCount: viewData.subtasks ? viewData.totalSubtaskCount - viewData.subtasks.length : 0,
            parentIssueKey: viewData.key,
            iconFont: 'jira-completed-task'
        });
    };

    DetailsFieldRenderer.renderSubtaskAsHtmlNode = function(tab) {
        var filteredSubtaskMessage = '';
        if (tab.filteredSubtaskCount > 0) {
            filteredSubtaskMessage = AJS.I18n.getText('gh.rapid.detail.view.subtasks.filtered',
                tab.filteredSubtaskCount,
                '<a href="' + GH.Ajax.CONTEXT_PATH + '/browse/' + tab.parentIssueKey + '#view-subtasks">',
                '</a>',
                '<a href="' + GH.Ajax.CONTEXT_PATH + '/secure/RapidView.jspa?tab=filter&rapidView=' + RapidViewConfig.currentData.data.id + '">'
            );
        }

        // we need to know if there are status categories to change the centering of status table column
        var firstSubtask = _.first(tab.subtasks);
        var statusCategoriesAvailable = firstSubtask && firstSubtask.status.statusCategory;

        return AJS.$(GH.tpl.detailview.renderSubtasks({
            subtasks: tab.subtasks,
            canCreateSubtask: tab.canCreateSubtask,
            showActionsColumn: tab.showActionsColumn,
            showTrackingField: tab.showTrackingField,
            trackingFieldSum: tab.trackingFieldSum,
            filteredSubtaskCount: tab.filteredSubtaskCount,
            filteredSubtaskMessage: filteredSubtaskMessage,
            statusCategoriesAvailable: statusCategoriesAvailable
        }));
    };

    DetailsFieldRenderer.renderSections = function(tab, tabPane) {
        var $tabTitle = AJS.$(GH.tpl.detailview.renderTabTitle({tabTitle: tab.title}));
        tabPane.append($tabTitle);
        _.each(tab.sections, function(section) {
            // render each section and append to above container
            this.renderSection(tabPane, section, tab.viewData);
        }.bind(this));
    };

    /**
     * Renders an entire 'module' (grouping) of fields
     * @param tabPane the tab to append this section to
     * @param section the details of the section (including title and fields)
     * @param viewData the view data of the current issue
     */
    DetailsFieldRenderer.renderSection = function(tabPane, section, viewData) {
        // render the module structure
        var $sectionDom = AJS.$(GH.tpl.detailview.renderTabSection({
            renderTitle : !_.isUndefined(section.title),
            title : section.title
        }));

        var sectionFieldsDom;
        if (section.fields) {
            if(section.fields.length === 0) {
                return;
            }

            // get a reference to the container for fields within this module
            sectionFieldsDom = $sectionDom.find('.ghx-tab-section-content');
            _.each(section.fields, function(field) {
                // render each field and append it to the above container
                var fieldDom = this.renderField(viewData, field);
                sectionFieldsDom.append(fieldDom);
            }.bind(this));

            tabPane.append($sectionDom);

        } else if (section.issueLinks) {
            sectionFieldsDom = $sectionDom.find('.ghx-tab-section-content');
            //var xsrfToken = AJS.$('#atlassian-token').attr('content');
            var issueLinksHtml = GH.tpl.detailview.issuelinks.renderIssueLinks({
                issueLinkTypeEntries : section.issueLinks.issueLinkTypeEntries,
                canLink : section.issueLinks.canLink/*,
                 atl_token : xsrfToken*/
            });
            sectionFieldsDom.replaceWith(issueLinksHtml);

            tabPane.append($sectionDom);
        }
    };

    /**
     * Renders a single field for Definition List usage
     */
    DetailsFieldRenderer.renderField = function(viewData, field) {
        var $fieldDom = AJS.$(GH.tpl.detailview.renderFieldStructure({issueDetails: viewData, field: field}));
        var fieldContent = this.renderFieldContent(field);
        $fieldDom.find('dd').html(fieldContent);
        return $fieldDom;
    };

    /**
     * Renders a single field with no wrapping
     */
    DetailsFieldRenderer.renderFieldBasic = function(viewData, field) {
        var fieldContent = this.renderFieldContent(field);
        return fieldContent;
    };

    DetailsFieldRenderer.renderLogWorkTrigger = function(viewData, detailHead, isTriggerShown) {
        var $logWorkContainer = detailHead.find('#ghx-log-work-container');
        if (isTriggerShown) {
            $logWorkContainer.css({'display':''});
        } else {
            $logWorkContainer.css({'display':'none'});
        }
    };

    /**
     * Renders the issue key
     */
    DetailsFieldRenderer.renderJiraLink = function(viewData, field, linkRoot) {
        var $linkDom = linkRoot.filter('.js-view-in-jira').add(linkRoot.find('.js-view-in-jira'));//filter also includes root element in matched set
        var data = {issueKey: field.text, eventName: 'gh.issueaction.issuedetail.open.full.issue'};
        if (field.id === PARENT_ISSUE_KEY_ID) {
            _.extend(data, {
                description: '...',
                skipTitle: true,
                eventName: 'gh.issueaction.issuedetail.open.full.issue.parent'
            });
        }
        $linkDom.find('dd').html(GH.tpl.detailview.renderIssueLink(data));
        // Trigger analytics event [SW-2193]
        $linkDom.click(function () {
            AJS.trigger('analytics', {name: data.eventName});
        });
    };

    /**
     * Ask the renderer to provide the markup for the given field
     */
    DetailsFieldRenderer.renderFieldContent = function(field) {
        var fieldContent;

        // handle description specially for now, as we have to show a different non-generic empty message (MiR)
        if (field.id === 'description') {
            fieldContent = GH.tpl.detailview.renderDescriptionHtmlView({field: field});
            return fieldContent;
        }

        // Estimate fields
        if (field.id === RapidViewConfig.currentData.data.estimationStatistic.fieldId) {
            fieldContent = GH.tpl.detailview.renderTextEstimateView({field: field});
            return fieldContent;
        }
        if (field.id === RapidViewConfig.currentData.data.trackingStatistic.fieldId) {
            fieldContent = GH.tpl.detailview.renderTextEstimateView({field: field});
            return fieldContent;
        }
        // aggregatetimeestimate should always be treated as an estimate.
        if (field.id === 'aggregatetimeestimate' && 'timeestimate' === RapidViewConfig.currentData.data.trackingStatistic.fieldId) {
            fieldContent = GH.tpl.detailview.renderTextEstimateView({field: field});
            return fieldContent;
        }

        switch (field.renderer) {
            case 'html':
                fieldContent = GH.tpl.detailview.renderHtmlView({field: field});
                break;
            case 'date':
                fieldContent = GH.tpl.detailview.renderDateView({field: field});
                break;
            case 'epiclink':
                fieldContent = GH.tpl.field.epicLink.renderEpicLink({field: field, removableIfAllowed: true});
                break;
            case 'status':
                fieldContent = GH.tpl.status.renderText({status: field.statusEntry});
                break;
            case 'text':
            /* falls through */
            default:
                fieldContent = GH.tpl.detailview.renderTextView({field: field});
                break;
        }
        return fieldContent;
    };

    return DetailsFieldRenderer;
});