/* global AJS */
define('jira-agile/rapid/ui/detail/details-view-scroll-tracker', [
    'underscore',
    'jquery',
    'require'
], function (_,
             $,
             require) {
    'use strict';

    var EVENT_NAMESPACE = "gh.issueaction.issuedetail";

    var GlobalEvents = require('jira-agile/rapid/global-events');
    var DetailsViewScroller = require('jira-agile/rapid/ui/detail/details-view-scroll-tracker/scroller');
    var Storage;

    GlobalEvents.on('pre-initialization', function () {
        Storage = GH.storage;
    });

    var $detailsNavContent;
    var $detailsNavMenu;

    var scrollTrackingCalculationNeeded;
    var scrollTrackingInterval;
    var savedScrollPosition;
    var lastIssueId;
    var onSelectedNavChanged;
    var overrideCurrentPosition;

    var DetailsViewScrollTracker = {
        /**
         * @param params.overrideTab force the tracker to load at the given tab
         * @param params.onSelectedNavChanged callback for when the selected nav section has changed.
         */
        init: init,

        /**
         * @param params.issueId the id of the issue the scroll tracker is tracking.
         *  The sectionId is given as a parameter.
         */
        afterDetailsViewLoaded: afterDetailsViewLoaded,

        /**
         * Sets the selected tab that should be loaded the next time init is called.
         * @param id of the tab.
         */
        setSelectedTab: setSelectedTab,
        _testing_only: {
            navMenuClick: navMenuClickHandler
        }
    };

    function init(params) {
        $detailsNavContent = $('#js-detail-nav-content');
        $detailsNavMenu = $('.ghx-detail-nav-menu');
        onSelectedNavChanged = _.isFunction(params.onSelectedNavChanged) ? params.onSelectedNavChanged : function() {};

        DetailsViewScroller.init({
            beforeScroll: stopScrollTracking,
            afterScroll: startScrollTracking
        });

        scrollTrackingCalculationNeeded = true;
        startScrollTracking();

        $detailsNavMenu.delegate('a', {
            'click.scrollTracking': navMenuClickHandler
        });

        // Add event listener for successful attachment message's link so we scroll to the attachment module when clicked
        $(document).on('click', '.aui-message-success a[href="' + AJS.Meta.get('viewissue-permlink') + '"]', function (event) {
            event.preventDefault();
            goToSection('attachmentmodule', true);
        });
    }

    function afterDetailsViewLoaded(params) {
        if (params.issueId !== lastIssueId || overrideCurrentPosition) {
            goToSection(getStoredSectionId());
            overrideCurrentPosition = false;
        } else if (savedScrollPosition >= 0) {
            goToPosition(savedScrollPosition);
        }
        lastIssueId = params.issueId;

    }

    function setSelectedTab(id) {
        setStoredTabId(id);
        overrideCurrentPosition = true;
    }

    function navMenuClickHandler(evt) {
        evt.preventDefault();
        var sectionId = this.hash.replace('#', '');
        goToSection(sectionId, true);
        triggerAnalyticsEventForAction('go.to.section', {sectionId: sectionId});
    }

    function goToSection(sectionId, animate) {
        var $section = $('#' + sectionId);

        if ($section.length > 0) {
            selectNavItem(sectionId);
            savedScrollPosition = DetailsViewScroller.scrollToSection(sectionId, animate)
        }
    }

    function goToPosition(position) {
        DetailsViewScroller.scrollTo(position);
        var currentSection = DetailsViewScroller.getCurrentSection();
        if (currentSection) {
            selectNavItem(currentSection);
        }
    }

    function startScrollTracking() {
        $detailsNavContent.bind({
            'scroll.scrollTracking': function () {
                scrollTrackingCalculationNeeded = true;
            }
        });

        scrollTrackingInterval = setInterval(function () {
            scrollTrackingCycle();
        }, 40);
    }

    /**
     * Watches the current scroll position, and updates the current scroll position to match.
     */
     function scrollTrackingCycle() {
        if (scrollTrackingCalculationNeeded) {
            scrollTrackingCalculationNeeded = false;
            savedScrollPosition = DetailsViewScroller.getScrollPosition();
            var currentSection = DetailsViewScroller.getCurrentSection();
            if (currentSection) {
                selectNavItem(currentSection);
            }
        }
    }

    function stopScrollTracking() {
        clearInterval(scrollTrackingInterval);
        $detailsNavContent.unbind('scroll.scrollTracking');
    }

    function selectNavItem(sectionId) {
        onSelectedNavChanged(sectionId);
        setStoredTabId(sectionId);
    }

    function triggerAnalyticsEventForAction(action, data) {
        AJS.trigger('analytics', {
            name: EVENT_NAMESPACE + '.' + action,
            data: data
        });
    }

    function getStoredSectionId() {
        return Storage.get('gh.detailTab') || null;
    }

    function setStoredTabId(tabId) {
        Storage.put('gh.detailTab', tabId);
    }

    return DetailsViewScrollTracker;
});
