define('jira-agile/dialog/inline', [
    '_'
], function (_) {

    var dialog = {
        /**
         * Creates AJS.InlineDialog but first ensures there is no existing dialog lingering around
         *
         * @param id dialog ID
         * @param trigger the trigger selector
         * @param renderFn the function to render the dialog
         * @param opts extra options to go to AJS.InlineDialog
         * @return the dialog created
         */
        create: function (id, trigger, renderFn, opts) {
            opts = opts || {};

            // first remove the existing dialog element from the DOM
            GH.InlineDialog.remove(id);

            // JAGDEV-329 - If the trigger is a button or a link, it will capture focus and stop propagation when
            // the Esc key is first pressed, preventing the dialog from being closed by pressing Esc.
            opts = GH.InlineDialog._addBlurTriggerOverride(opts, trigger);

            // instantiate
            return AJS.InlineDialog(AJS.$(trigger), id, renderFn, opts);
        },

        /**
         * Remove an InlineDialog's element from the DOM
         * @param id
         */
        remove: function (id) {
            AJS.$("#inline-dialog-" + id).remove();
        },

        /**
         * Copied from ChartUtils.
         *
         * @param dialogId the ID of the dialog
         * @param opts some options around checking whether or not to close the dialog when handling click events
         */
        registerInlineDialogHacks: function (dialogId, opts) {
            var disregardClickSelectors = (opts && opts.disregardClickSelectors) || [];
            var validateClickToCloseFn = (opts && opts.validateClickToCloseFn) || false;

            // hacks to handle closing of the layer. We need to attach a close evaluation function that does NOT close if we either click on the dialog
            // or on one of the calendars.
            AJS.$(document).bind("showLayer", function (e, type, inlineLayer) {
                GH.log(e.type + " handled", GH.Logger.Contexts.event);

                if (type === "inlineDialog" && inlineLayer.id === dialogId) {
                    // handle InlineLayer out of zone click handler
                    inlineLayer._validateClickToClose = function (e) {
                        var currentDialogId = this.id;

                        GH.log("Validating whether or not to close the InlineDialog based on a click", GH.Logger.Contexts.event);

                        // specific selectors that when clicked should not close the dialog
                        for (var i = 0; i < disregardClickSelectors.length; i++) {
                            var $target = AJS.$(e.target);
                            if ($target.parents(disregardClickSelectors[i]).length > 0) {
                                return false;
                            }
                            if ($target.is(disregardClickSelectors[i])) {
                                return false;
                            }
                        }

                        // clicks inside the dialog
                        if (AJS.$(e.target).parents('#inline-dialog-' + currentDialogId).length > 0) {
                            return false;
                        }

                        // if we specified a custom validate function, and that function says don't close, then don't close
                        if (validateClickToCloseFn && !validateClickToCloseFn(e)) {
                            return false;
                        }

                        return true;
                    };
                    // handle InlineDialog out of zone click handler - inline layer is good enough plus this guy doesn't allow us any overwriting.
                    // --> Kill
                    AJS.$('body').unbind('click.' + inlineLayer.id + '.inline-dialog-check');
                }
            });
        },

        /**
         * Prevent a click inside an inline dialog from closing the dialog.
         *
         * @param dialogId The dialog
         */
        preventCloseOnClick: function (dialogId) {
            AJS.$(document).on('click', '#inline-dialog-' + dialogId, function (e) {
                e.stopPropagation();
            });
        },

        /**
         * Augments dialog options to blur the trigger when the dialog is shown.
         *
         * @param opts
         * @param trigger selector
         * @return copy of opts with added behaviour
         * @private
         */
        _addBlurTriggerOverride: function (opts, trigger) {
            var oldUpfrontCallback = opts.upfrontCallback;

            return _.defaults({
                upfrontCallback: function () {
                    AJS.$(trigger).blur();

                    if (_.isFunction(oldUpfrontCallback)) {
                        oldUpfrontCallback.apply(this, arguments);
                    }
                }
            }, opts);
        }
    };

    return dialog;
});

AJS.namespace('GH.InlineDialog', null, require('jira-agile/dialog/inline'));