define('jira-agile/rapid/ui/epic/epic-controller', ['require'], function (require) {

    var logger = require('jira/util/logger');
    var _ = require('underscore');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var analytics = require('jira/analytics');
    var AnalyticsHelper = require('jira-agile/rapid/ui/plan/analytics-helper');

    var IGNORE_EPICS_ON_EPIC_ASSIGN = true;
    var IGNORE_SUBTASKS_ON_EPIC_ASSIGN = true;

    /**
     * Epic view component.
     */
    var EpicController = {};

    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    // TODO: change code using second to use first
    EpicController.analytics = new AnalyticsTracker("gh.rapidboard.epics");
    EpicController.analytics2 = new AnalyticsTracker("gh.rapidboard.epics", "epicCount", "");

    EpicController.sendAnalytics = function (eventKey, data, skipDefaults) {
        if (!eventKey) {
            throw new Error("You must specify a category");
        }
        var eventName = 'jira-software.' + GH.RapidBoard.State.getBoardType() + '.plan.epics.' + eventKey;
        var defaultData = AnalyticsHelper.baseEventData();

        analytics.send({
            name: eventName,
            properties: skipDefaults ? data : _.extend(defaultData, data)
        });
    };

    EpicController.model = undefined;

    EpicController.init = function () {
        // handle new issue created (from Quick Create) -- for analytics
        AJS.$(GH).bind('issueCreated', EpicController.handleIssueCreated);

        EpicController.model = new GH.EpicModel();

        // initialize the epic view
        GH.EpicView.init();
    };

    /**
     * @return {String} the key of the epic currently active as a filter
     */
    EpicController.getFilteredEpicKey = function () {
        return GH.BoardState.getPerViewValue('planSelectedEpic', null);
    };

    EpicController.setFilteredEpicKey = function (epicKey, skipAnalytics) {
        GH.BoardState.setPerViewValue('planSelectedEpic', epicKey);

        if (!skipAnalytics) {
            EpicController.sendAnalytics(epicKey ? "epicFilterActive" : "epicFilterInactive");
        }
    };

    EpicController.getEpicModel = function () {
        return EpicController.model;
    };

    /**
     * Sets the epic data
     *
     * @param epicData
     */
    EpicController.setEpicData = function (epicData) {
        EpicController.model.setData(epicData);
    };

    /**
     * Validates the currently selected epic, removes the selection and updates the url if invalid
     */
    EpicController.validateSelectedEpic = function () {
        // apply epic filter from loaded state if present
        var epicList = EpicController.getEpicModel().getEpicList();
        var filteredEpicKey = EpicController.getFilteredEpicKey();

        // is the selection valid?
        var isEpicKeyValid = _.isNull(filteredEpicKey) || GH.PlanIssueListFiltering.isNoneEpicFilterKey(filteredEpicKey) && epicList.getNumVisibleIssues() > 0 || epicList.isIssueValid(filteredEpicKey);

        if (!isEpicKeyValid) {
            // remove the epic filtering key from the board settings if it doesn't exist
            EpicController.setFilteredEpicKey(null);
            GH.RapidBoard.UrlState.replaceState();
        }
    };

    EpicController.registerEpicsCount = function (epicData) {
        // track number of epics on board

        AJS.trigger('analytics', {
            name: 'jira-software.board.number.of.epics',
            data: {
                boardId: GH.RapidBoard.State.getRapidViewId(),
                epicsOnBoard: epicData.epics.length
            }
        });
    };

    EpicController.reorderEpics = function (epicKeys, prevEpicKey, nextEpicKey) {
        EpicController.model.getEpicList().reorderIssues(epicKeys, prevEpicKey, nextEpicKey);
        EpicController.sendAnalytics("epicsReordered");
    };

    EpicController.handleIssueCreated = function (event, data) {
        // if Analytics are disabled, don't bother
        if (!GH.AnalyticsEnabled) {
            return;
        }

        // get the Epic Label custom field ID
        var epicLabelFieldId = GH.EpicConfig.getEpicLabelFieldId();
        if (!_.isString(epicLabelFieldId) || epicLabelFieldId.length === 0) {
            return;
        }

        // query JIRA for all created issues so we can get more information on them
        _.each(data.issues, function (issue) {
            GH.Ajax.get({
                bareUrl: GH.Ajax.CONTEXT_PATH + '/rest/api/2/issue/' + issue.issueKey + '?fields=issuetype,' + epicLabelFieldId
            }, "getEpicLabelLength").done(function (issue) {
                // only care about Epics
                if (issue.fields.issuetype.name !== "Epic") {
                    return;
                }

                var epicLabel = issue.fields[epicLabelFieldId] || "";

                // fire analytics
                EpicController.analytics.trigger("epicLabelLength", "length", epicLabel.length);
                EpicController.sendAnalytics('epicCreated');
            });
        });
    };

    /**
     * Adds issues to an epic
     */
    EpicController.addIssuesToEpic = function (epicKey, issueKeys) {
        // update the epics before going to the server
        // set the epic field value for the issues in question
        // avoid adding epic issues to epics
        var issues = [];
        _.each(issueKeys, function (issueKey) {
            var issueData = GH.BacklogModel.getIssueData(issueKey);
            if (issueData && issueData.typeId !== GH.EpicConfig.getEpicIssueTypeId()) {
                issueData.epic = epicKey;
                issues.push(issueData);
            }
        });

        // then update on the server
        EpicController.addIssuesToEpicOnServer(epicKey, issueKeys).done(function (result) {
            GH.Notification.showSuccess(getProgressiveMessageFor('gh.epic.issues.added', issues));

            _.each(issues, function (issue) {
                AJS.$(GH).trigger('issueUpdated', { issueId: issue.id, source: 'detailView' });
            });

            if (GH.PlanController.isActive()) {
                // update the epics data (earlier the data on the server wouldn't be correct)
                GH.BacklogController.updateEpicData();

                // Refresh the detail view
                EpicController._reloadDetailsView(GH.PlanController);
            }

            EpicController.analytics.trigger("addIssues", "issueCount", issueKeys.length);
            EpicController.sendAnalytics("addIssuesToEpic", { issueCount: issueKeys.length });
        }).fail(function () {
            // something didn't quite work out. reload to get back into a good state
            GH.BacklogController.loadData();
        });
    };

    EpicController.addIssuesToEpicOnServer = function (epicKey, issueKeys) {
        var addToEpicRequest = {
            ignoreEpics: IGNORE_EPICS_ON_EPIC_ASSIGN,
            ignoreSubtasks: IGNORE_SUBTASKS_ON_EPIC_ASSIGN,
            issueKeys: issueKeys
        };

        return GH.Ajax.put({
            url: '/epics/' + epicKey + '/add',
            data: addToEpicRequest,
            errorContextMap: {
                'soft-error': function softError(response) {
                    'use strict';

                    response && GH.Notification.showErrors({ errors: response.errors }, true, {
                        autoHide: true,
                        showTitle: false
                    });
                }
            }
        });
    };

    /**
     * It iterates through the provided keys array and clear the reference to the epics.
     *
     * @param {Array} issueKeys keys of the key to remove from associated epics
     */
    EpicController.removeIssuesFromAssociatedEpics = function (issueKeys) {
        var issues = [];
        if (GH.PlanController.isActive()) {
            issues = GH.BacklogModel.removeEpicFromIssues(issueKeys);
            if (issues.length === 0) {
                return;
            }
        } else if (GH.WorkController.isActive()) {
            issues = GH.GridDataController.getModel().removeEpicFromIssues(issueKeys);
            if (issues.length === 0) {
                return;
            }
        }

        var removeFromEpicRequest = {
            issueKeys: issueKeys
        };

        GH.Ajax.put({
            url: '/epics/remove',
            data: removeFromEpicRequest
        }).done(function (result) {
            GH.Notification.showSuccess(getProgressiveMessageFor('gh.epic.issues.removed', issues));

            if (!_.isEmpty(issues)) {
                _.each(issues, function (issue) {
                    AJS.$(GH).trigger('issueUpdated', { issueId: issue.id, source: 'detailView' });
                });
            }

            // the stuff below is done outside of the issueUpdated event so it only happens once rather than per issue
            if (GH.PlanController.isActive()) {
                // reload the epics data
                GH.BacklogController.updateEpicData();

                // reload the detail view
                EpicController._reloadDetailsView(GH.PlanController);
            } else if (GH.WorkController.isActive()) {
                // reload the detail view
                EpicController._reloadDetailsView(GH.WorkController);
                // if epic swimlanes are active, reload swimlanes
                if (GH.GridDataController.getModel().getSwimlaneStrategy() === "epic") {
                    GH.WorkController.reload();
                }
            }

            EpicController.analytics.trigger("removeIssues", "issueCount", issueKeys.length);
            EpicController.sendAnalytics("removeIssues", { issueCount: issueKeys.length });
        });
    };

    /**
     * Returns a message which depends on how many issues were altered.
     * It uses given i18n key and passes to getText number of issues and up to 2 issue keys:
     *  AJS.I18n.getText(i18nKey, issues.length, issue1Key, issue2Key);
     *
     * Take a look into impl which i18n keys are supported.
     *
     * @param i18nKey {string}
     * @param issues {IssueData[]} list of issue objects
     * @returns {string} progressive message
     */
    function getProgressiveMessageFor(i18nKey, issues) {
        issues = issues || [];
        var i18n = [];
        var ignoredSubtasksCount = void 0;

        // Do we need to exclude subtasks?
        if (IGNORE_SUBTASKS_ON_EPIC_ASSIGN) {
            // cachce current issues count
            ignoredSubtasksCount = issues.length;
            // filter out unwanted subtasks (we'll need the "clean" list later)
            issues = _.reject(issues, function (issue) {
                return !!issue.parentId;
            });
            // substract new list's count. Difference will be equal to the count of filtered out subtasks
            ignoredSubtasksCount -= issues.length;
        }

        var length = issues.length;
        var issue1Key = length >= 1 ? issues[0].key : undefined;
        var issue2Key = length >= 2 ? issues[1].key : undefined;

        // because of JIRA's i18n build-time transformer i18n keys have to be explicitly written
        switch (i18nKey) {
            case 'gh.epic.issues.removed':
                logger.trace('gh.epic.issues.removed', issues.length, issue1Key, issue2Key);
                i18n.push(AJS.I18n.getText('gh.epic.issues.removed', issues.length, issue1Key, issue2Key));
                break;
            case 'gh.epic.issues.added':
                logger.trace('gh.epic.issues.added', issues.length, issue1Key, issue2Key);
                i18n.push(AJS.I18n.getText('gh.epic.issues.added', issues.length, issue1Key, issue2Key));
                break;
        }

        if (!!ignoredSubtasksCount) {
            i18n.push(AJS.I18n.getText("gh.epic.subtasks.skipped"));
        }

        return i18n.join("<br /><br />");
    }

    EpicController.updateEpicAsDone = function (epicKey) {
        var xhr = EpicController.updateEpicField(epicKey, GH.EpicConfig.getEpicStatusFieldId(), GH.EpicConfig.getEpicStatusDoneValueId());
        xhr.done(function () {
            // analytics
            EpicController.analytics.trigger("markEpicAsDone");

            // show success message
            GH.EpicView.showEpicMarkedAsDoneMessage();

            // if epic was selected, de-select it and clear the active filter on the backlog
            if (EpicController.getFilteredEpicKey() === epicKey) {
                EpicController.setFilteredEpicKey(null);

                GH.BacklogController.updateEpicFiltering(false);

                // replace state instead of push because the previous state is now invalid
                GH.RapidBoard.State.replaceState();
            }

            // reload epics data and re-render
            GH.BacklogController.updateEpicData();
        });
    };

    EpicController.updateEpicField = function (epicKey, fieldId, newValue, successFn, errorFn) {

        return GH.Ajax.put({
            url: '/xboard/issue/update-field.json',
            data: {
                'issueIdOrKey': epicKey,
                'fieldId': fieldId,
                'newValue': newValue
            }
        }).done(successFn).fail(errorFn);
    };

    /**
     * Applies the filtering to the epics based on the currently active version filter. If no version filter is active, no filtering will be applied.
     * If the version filtering reveals that the selected epic will be filtered, the current epic filter will be cleared
     */
    EpicController.applyVersionFiltering = function () {
        if (GH.VersionController.hasFilteredVersion()) {
            var epicsFilteredForVersion = GH.BacklogModel.getEpicsFilteredForVersion(GH.VersionController.getFilteredVersionId());
            var epicFilterKey = EpicController.getFilteredEpicKey();
            // if selected epic will be filtered, deselect it
            if (_.contains(epicsFilteredForVersion.filtered, epicFilterKey)) {
                GH.EpicView.deselectAllEpics();
                EpicController.setFilteredEpicKey(null);
                GH.BacklogController.updateEpicFiltering(false);
                GH.RapidBoard.State.pushState();
            }
            GH.EpicView.setFilteredStyle(epicsFilteredForVersion.filtered);
        } else {
            EpicController.clearVersionFiltering();
        }
    };

    /**
     * Clear the applied version filtering
     */
    EpicController.clearVersionFiltering = function () {
        GH.EpicView.clearFilteredStyle();
    };

    /**
     * Private function which centralize the logic of when we make a call to reload Details View with a reason.
     * @param controller
     * @private
     */
    EpicController._reloadDetailsView = function (controller) {
        if (GH.Features.EDITABLE_DETAIL_VIEW_ENABLED.isEnabled()) {
            var EditableDetailsViewReloadReason = require('jira-agile/rapid/ui/detail/inlineedit/details-view-reload-reason');
            controller.reloadDetailView(EditableDetailsViewReloadReason.VERSION_PANEL_CHANGED);
        } else {
            controller.reloadDetailView();
        }
    };

    return EpicController;
});