AJS.test.require(["com.pyxis.greenhopper.jira:gh-notification"], function() {
    'use strict';

    module('Notification', {
        setup: function () {
            this.sandbox = sinon.sandbox.create();
            let context = this.context = AJS.test.mockableModuleContext();

            this.mockJQuery();

            this.auiFlag = sinon.stub();
            context.mock('aui/flag', this.auiFlag);

            this.Notification = context.require('jira-agile/rapid/ui/notification');
        },
        teardown: function () {
            this.sandbox.restore();
        },
        mockJQuery: function () {
            this.jQueryMethods = {
                find: sinon.stub(),
                text: sinon.stub(),
                remove: sinon.stub(),
            };
            this.jQueryMethods.find.returns(this.jQueryMethods);

            this.$ = sinon.stub().returns(this.jQueryMethods);
            this.$.trim = sinon.stub();

            this.context.mock('jquery', this.$);
        },
        /**
         * @param {function} callback which will be called like that: `callback.call(this, message, title, opts)`
         *          whereas args come from "specs[].in" input data
         * @param {{name: string, in: object, out: object}[]} specs
         * @param {string} specs[].name name of the spec which will be printed for better readability
         * @param {{message: string, title: string, autoHide: bool, closeable: bool, showTitle: bool}} specs[].in
         *          input data which will be passed to callback
         * @param {{body: string, title: string, close: string, type: string}} specs[].out
         *          output data - "aui-flag" has to be called with those arguments
         */
        assertSpecs: function (callback, specs) {
            specs.forEach(function(spec) {
                ok(true, `Running spec: ${spec.name}`);
                let opts = {};
                if (spec.in.hasOwnProperty('autoHide')) {
                    opts.autoHide = spec.in.autoHide;
                }
                if (spec.in.hasOwnProperty('closeable')) {
                    opts.closeable = spec.in.closeable;
                }
                if (spec.in.hasOwnProperty('showTitle')) {
                    opts.showTitle = spec.in.showTitle;
                }
                if (spec.in.hasOwnProperty('preRendered')) {
                    opts.preRendered = spec.in.preRendered;
                }

                callback.call(this, spec.in.message, spec.in.title, opts);

                sinon.assert.calledOnce(this.auiFlag);
                sinon.assert.alwaysCalledWithExactly(this.auiFlag, {
                    body: spec.out.body,
                    title: spec.out.title,
                    close: spec.out.close,
                    type: spec.out.type
                });

                this.auiFlag.reset();
            }, this);
        }
    });


    test("showSuccess should call `aui-flag` with proper args", function () {
        let specs = [
            {
                name: 'default',
                in: {message: 'test'},
                out: {body: 'test', title: undefined, close: 'auto', type: 'success'}
            },
            {
                name: 'no support for title',
                in: {message: 'msg', title: 'no-title', showTitle: true},
                out: {body: 'msg', title: null, close: 'auto', type: 'success'}
            },
            {
                name: 'HTML in the message',
                in: {message: 'msg <b>strong</b>'},
                out: {body: 'msg <b>strong</b>', title: undefined, close: 'auto', type: 'success'}
            },
            {
                name: 'Not closeable, without auto hide',
                in: {message: 'msg', autoHide: false, closeable: false},
                out: {body: 'msg', title: undefined, close: 'never', type: 'success'}
            },
            {
                name: 'Not closeable, with auto hide',
                in: {message: 'msg', autoHide: true, closeable: false},
                out: {body: 'msg', title: undefined, close: 'auto', type: 'success'}
            },
            {
                name: 'Closeable, without auto hide',
                in: {message: 'msg', autoHide: false, closeable: true},
                out: {body: 'msg', title: undefined, close: 'manual', type: 'success'}
            },
            {
                name: 'Closeable, with auto hide',
                in: {message: 'msg', autoHide: true, closeable: true},
                out: {body: 'msg', title: undefined, close: 'auto', type: 'success'}
            },
        ];

        this.assertSpecs((message, title, opts) => {
            this.Notification.showSuccess(message, opts);
        }, specs);
    });

    test("showWarning should call `aui-flag` with proper args", function () {
        let specs = [
            {
                name: 'default',
                in: {message: 'test'},
                out: {body: 'test', title: undefined, close: 'auto', type: 'warning'}
            },
            {
                name: 'no support for title',
                in: {message: 'msg', title: 'ignored-title', showTitle: true},
                out: {body: 'msg', title: undefined, close: 'auto', type: 'warning'}
            },
            {
                name: 'HTML in the message',
                in: {message: 'msg <b>strong</b>'},
                out: {body: 'msg <b>strong</b>', title: undefined, close: 'auto', type: 'warning'}
            },
            {
                name: 'no support for opts',
                in: {message: 'msg', autoHide: false, closeable: false},
                out: {body: 'msg', title: undefined, close: 'auto', type: 'warning'}
            },
        ];

        this.assertSpecs((message, title, opts) => {
            this.Notification.showWarning(message);
        }, specs);
    });

    test("showWarnings should call `aui-flag` with proper args (when preRendered: true, so `renderErrorMessages` is not used)", function () {
        let specs = [
            {
                name: 'default',
                in: {message: 'test', preRendered: true},
                out: {body: 'test', title: undefined, close: 'auto', type: 'warning'}
            },
            {
                name: 'default title',
                in: {message: 'msg', title: 'ignored-title', showTitle: true, preRendered: true},
                out: {body: 'msg', title: 'gh.error.warningtitle', close: 'auto', type: 'warning'}
            },
            {
                name: 'HTML in the message',
                in: {message: 'msg <b>strong</b>', preRendered: true},
                out: {body: 'msg <b>strong</b>', title: undefined, close: 'auto', type: 'warning'}
            },
            {
                name: 'Not closeable, without auto hide',
                in: {message: 'msg', autoHide: false, closeable: false, preRendered: true},
                out: {body: 'msg', title: undefined, close: 'never', type: 'warning'}
            },
            {
                name: 'Not closeable, with auto hide',
                in: {message: 'msg', autoHide: true, closeable: false, preRendered: true},
                out: {body: 'msg', title: undefined, close: 'auto', type: 'warning'}
            },
            {
                name: 'Closeable, without auto hide',
                in: {message: 'msg', autoHide: false, closeable: true, preRendered: true},
                out: {body: 'msg', title: undefined, close: 'manual', type: 'warning'}
            },
            {
                name: 'Closeable, with auto hide',
                in: {message: 'msg', autoHide: true, closeable: true, preRendered: true},
                out: {body: 'msg', title: undefined, close: 'auto', type: 'warning'}
            },
        ];

        this.assertSpecs((message, title, opts) => {
            this.Notification.showWarnings(message, opts);
        }, specs);
    });

    test("showError should call `aui-flag` with proper args", function () {
        let specs = [
            {
                name: 'default',
                in: {message: 'test', title: 'a title'},
                out: {body: 'test', title: 'a title', close: 'manual', type: 'error'}
            },
            {
                name: 'support for no title',
                in: {message: 'msg', title: 'ignored-title', showTitle: false},
                out: {body: 'msg', title: undefined, close: 'manual', type: 'error'}
            },
            {
                name: 'HTML in the message',
                in: {message: 'msg <b>strong</b>', title: 'a title'},
                out: {body: 'msg <b>strong</b>', title: 'a title', close: 'manual', type: 'error'}
            },
            {
                name: 'Not closeable, without auto hide',
                in: {message: 'msg', autoHide: false, closeable: false},
                out: {body: 'msg', title: undefined, close: 'never', type: 'error'}
            },
            {
                name: 'Not closeable, with auto hide',
                in: {message: 'msg', autoHide: true, closeable: false},
                out: {body: 'msg', title: undefined, close: 'auto', type: 'error'}
            },
            {
                name: 'Closeable, without auto hide',
                in: {message: 'msg', autoHide: false, closeable: true},
                out: {body: 'msg', title: undefined, close: 'manual', type: 'error'}
            },
            {
                name: 'Closeable, with auto hide',
                in: {message: 'msg', autoHide: true, closeable: true},
                out: {body: 'msg', title: undefined, close: 'auto', type: 'error'}
            },
        ];

        this.assertSpecs((message, title, opts) => {
            this.Notification.showError(title, message, opts.closeable, opts);
        }, specs);
    });

    test("showErrors should call `aui-flag` with proper args (`renderErrorMessages` is always used)", function () {
        let messages = {errors: [
            {
                message: 'a msg <strong>with autoescape</strong>',
                noAutoescape: false
            },
            {
                message: 'a msg <strong>without autoescape</strong>',
                noAutoescape: true
            },
        ]};
        let renderedMessages = '<ul><li>a msg &lt;strong&gt;with autoescape&lt;/strong&gt;</li><li>a msg <strong>without autoescape</strong></li></ul>';

        let specs = [
            {
                name: 'default',
                in: {message: messages, title: 'ignored-title'},
                out: {body: renderedMessages, title: 'gh.error.errortitle', close: 'manual', type: 'error'}
            },
            {
                name: 'support for no title',
                in: {message: messages, title: 'no-title', showTitle: false},
                out: {body: renderedMessages, title: undefined, close: 'manual', type: 'error'}
            },
            {
                name: 'Not closeable, without auto hide',
                in: {message: messages, autoHide: false, closeable: false},
                out: {body: renderedMessages, title: 'gh.error.errortitle', close: 'never', type: 'error'}
            },
            {
                name: 'Not closeable, with auto hide',
                in: {message: messages, autoHide: true, closeable: false},
                out: {body: renderedMessages, title: 'gh.error.errortitle', close: 'auto', type: 'error'}
            },
            {
                name: 'Closeable, without auto hide',
                in: {message: messages, autoHide: false, closeable: true},
                out: {body: renderedMessages, title: 'gh.error.errortitle', close: 'manual', type: 'error'}
            },
            {
                name: 'Closeable, with auto hide',
                in: {message: messages, autoHide: true, closeable: true},
                out: {body: renderedMessages, title: 'gh.error.errortitle', close: 'auto', type: 'error'}
            },
        ];

        this.assertSpecs((message, title, opts) => {
            this.Notification.showErrors(message, opts.closeable, opts);
        }, specs);
    });

    test("'clear' should remove all notifications", function () {
        let emptyStub = sinon.stub();
        this.$.withArgs('#aui-flag-container').returns({empty: emptyStub});

        this.Notification.clear();

        sinon.assert.calledOnce(emptyStub);
    });
});