/* globals
 * GH.Dialogs, GH.EpicView, GH.planOnboarding, GH.Ajax, GH.Logger, GH.RapidBoard, GH.VersionController,
 * GH.EpicController, GH.VersionView
 */

/**
 * Initialise the Backlog controller.
 * This module handles backlog page
 * @module jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:underscore
 * @requires module:jquery
 * @requires module:jira/util/formatter
 * @requires module:jira/util/browser
 * @requires module:jira-agile/rapid/ui/plan/backlog-model
 * @requires module:jira-agile/rapid/ui/plan/BacklogView
 * @requires module:jira-agile/rapid/ui/plan/backlog-selection-controller
 * @requires module:jira-agile/rapid/ui/plan/sprint-backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/sprint-controller
 * @requires module:jira-agile/rapid/ui/plan/sprint-view
 * @requires module:jira-agile/rapid/ui/plan/sprint-backlog-view
 * @requires module:jira-agile/rapid/ui/plan/sprint-config
 * @requires module:jira-agile/rapid/ui/plan/plan-rank-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-controls
 * @requires module:jira-agile/rapid/ui/plan/plan-view
 * @requires module:jira-agile/rapid/ui/plan/plan-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-issue-list-filtering
 */

define('jira-agile/rapid/ui/plan/backlog-controller', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var $ = require('jquery');
    var formatter = require('jira/util/formatter');
    var Browser = require('jira/util/browser');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogView = require('jira-agile/rapid/ui/plan/BacklogView');
    var BacklogSelectionController = require('jira-agile/rapid/ui/plan/backlog-selection-controller');
    var SprintBacklogController = require('jira-agile/rapid/ui/plan/sprint-backlog-controller');
    var SprintView = require('jira-agile/rapid/ui/plan/sprint-view');
    var SubtasksExpandingController = require('jira-agile/rapid/ui/plan/subtasks-expanding-controller');
    var SprintBacklogView = require('jira-agile/rapid/ui/plan/sprint-backlog-view');
    var SprintConfig = require('jira-agile/rapid/ui/plan/sprint-config');
    var PlanRankController = require('jira-agile/rapid/ui/plan/plan-rank-controller');
    var PlanControls = require('jira-agile/rapid/ui/plan/plan-controls');
    var PlanView = require('jira-agile/rapid/ui/plan/plan-view');
    var PlanController = require('jira-agile/rapid/ui/plan/plan-controller');
    var PlanIssueListFiltering = require('jira-agile/rapid/ui/plan/plan-issue-list-filtering');
    var SprintController = require('jira-agile/rapid/ui/plan/sprint-controller');
    var KanPlanFeatureService = require('jira-agile/rapid/ui/kanplan/kan-plan-feature-service');
    var EditableDetailsViewReloadReason = null;

    var BacklogController = {};

    BacklogController.visible = false;

    BacklogController.rapidViewData = undefined;
    BacklogController.rapidViewConfig = undefined;

    /**
     * Keep track of the last executed instant filter search function
     */
    BacklogController.lastExecutedSearchFilter = undefined;

    BacklogController.issueCreatedCallbacks = {};

    /**
     * Initialise the Backlog controller.
     */
    BacklogController.init = function () {
        BacklogSelectionController.init();
        SprintBacklogController.init();
        SprintController.init();
        PlanRankController.init();

        var $GH = $(GH);

        // listen to updates to an issue from the detail view.
        $GH.bind('issueUpdated', BacklogController.handleIssueUpdated);

        // handle new issue created
        $GH.bind('issueCreated', BacklogController.handleIssueCreated);

        // handle issues removed from sprint
        $GH.bind('issuesRemovedFromSprint', BacklogController.handleIssuesRemovedFromSprint);

        // handle sprint started
        $GH.bind(GH.Dialogs.StartSprintDialog.EVENT_SPRINT_STARTED, BacklogController.handleSprintStarted);

        // bind on the quick filter component's change event
        $(PlanControls.quickfilters).bind('quickFilterChange', function () {
            if (BacklogController.visible) {
                BacklogController.loadData();
            }
        });

        if (GH.Features.EDITABLE_DETAIL_VIEW_ENABLED.isEnabled()) {
            require('jira-agile/rapid/ui/detail/detail-view-resources').done(function () {
                EditableDetailsViewReloadReason = require('jira-agile/rapid/ui/detail/inlineedit/details-view-reload-reason');
            });
        }
    };

    /**
     * Sets the rapid view data.
     */
    BacklogController.setRapidView = function (rapidViewData) {
        BacklogController.rapidViewData = rapidViewData;
    };

    /**
     * Sets the rapid view CONFIG (DIFFERENT!)
     */
    BacklogController.setRapidViewConfig = function (rapidViewConfig) {
        BacklogController.rapidViewConfig = rapidViewConfig;
    };

    /**
     * Show the Backlog. Should only be called when we have a rapid view set.
     */
    BacklogController.show = function (callback) {
        if (!BacklogController.rapidViewData) {
            return false;
        }

        BacklogController.visible = true;

        BacklogController.loadData().done(callback);
    };

    /**
     * Hide the Backlog.
     */
    BacklogController.hide = function () {
        GH.EpicView.hide();
        BacklogController.visible = false;
    };

    /**
     * Load the data for the Backlog.
     */
    BacklogController.loadData = function (detailsViewReloadReason) {
        var params = {
            rapidViewId: BacklogController.rapidViewData.id,
            activeQuickFilters: PlanControls.quickfilters.getActiveQuickFilters(),
            selectedProjectKey: GH.RapidBoard.projectKey
        };

        if (_.isEmpty(params.activeQuickFilters)) {
            delete params.activeQuickFilters;
        }

        PlanView.showLoadingBacklog();
        GH.planOnboarding.refresh();

        var settings = {
            url: '/xboard/plan/backlog/data.json',
            data: params
        };

        var dataLoading = GH.Ajax.get(settings, 'backlogDataModel').done(BacklogController.processData.bind(null, detailsViewReloadReason));

        dataLoading.fail(function (response) {
            if (response.error && response.error.errors && response.error.errors.length) {
                var errorMesage = response.error.errors[0].message;
                if (errorMesage === formatter.I18n.getText('gh.boards.kanplan.error.feature.not.enabled')) {
                    Browser.reloadViaWindowLocation();
                }
            }

            // hide loading for the backlog
            PlanView.hideLoadingBacklog();

            // most likely got an error because Sprint Index Check failed
            PlanController.hide();
        });

        return dataLoading.promise();
    };

    /**
     * Process the data which was loaded.
     */
    BacklogController.processData = function (detailsViewReloadReason, data) {
        GH.log('loaded data for backlog', GH.Logger.Contexts.ajax);

        if (!BacklogController.visible) {
            return;
        }

        BacklogController.lastExecutedSearchFilter = undefined;

        // set whether or not we can rank based on custom field ID
        BacklogModel.setRankCustomFieldId(data.rankCustomFieldId);

        // set the estimation and tracking statistic
        BacklogModel.setEstimationStatistic(BacklogController.rapidViewConfig.estimationStatistic);
        BacklogModel.setTrackingStatistic(BacklogController.rapidViewConfig.trackingStatistic);

        // set the versions data
        GH.VersionController.setProjects(data.projects);
        GH.VersionController.setVersions(data.versionData);

        // set the epic data
        GH.EpicController.setEpicData(data.epicData);

        // set the issues, epics, open sprints and markers
        BacklogModel.setData(data);

        // set manage sprint permission
        BacklogModel.setCanManageSprints(data.canManageSprints);

        // set manage sprint permission
        BacklogModel.setCanCreateIssue(data.canCreateIssue);

        BacklogModel.setSupportsPages(data.supportsPages);

        // validate the issue selection
        BacklogSelectionController.validateCurrentSelection();
        GH.VersionController.validateSelectedVersion();
        GH.EpicController.validateSelectedEpic();
        BacklogSelectionController.selectAllCollapsedSubtasks();

        // draw
        GH.VersionView.show();
        GH.EpicView.show();

        // set the issue filters
        BacklogController.handleVersionFiltering();
        BacklogController.handleEpicFiltering();

        // draw the view
        BacklogView.draw();

        // analytics stuff
        GH.VersionController.registerVersionsCount();
        GH.EpicController.registerEpicsCount(data.epicData);

        // re-execute the local search if necessary
        PlanControls.searchFilter.clearLastSearch();
        PlanControls.searchFilter.searchQuery();

        // hide loading for the backlog
        PlanView.hideLoadingBacklog();
        // If no epics and versions then hide the left border in #ghx-backlog so there is no double line shown
        PlanView.updateEpicsAndVersionsColumnBorder();

        GH.RapidBoard.ViewController.updateContentContainer();

        // update the detail view
        PlanController.updateDetailsView(BacklogSelectionController.getSelectedIssueKey(), null, detailsViewReloadReason);

        // scroll to the selected issue
        BacklogView.updateIssueSelectionState(true);
    };

    /**
     * Clears epic filter and corrects the URL state when epics panel is disabled
     */
    BacklogController.handleVersionFiltering = function () {
        if (KanPlanFeatureService.isEpicsAndVersionsEnabled()) {
            BacklogController.updateVersionFiltering(true);
        } else {
            GH.VersionController.setFilteredVersionId(null);
            PlanController.setVersionsColumnVisible(false);
            // replace state so that we don't keep the faulty state in history
            GH.RapidBoard.State.replaceState();
        }
    };

    /**
     * Clears epic filter and corrects the URL state when epics panel is disabled
     */
    BacklogController.handleEpicFiltering = function () {
        if (PlanController.isEpicsPanelEnabled()) {
            BacklogController.updateEpicFiltering(true);
        } else {
            GH.EpicController.setFilteredEpicKey(null);
            PlanController.setEpicsColumnVisible(false);
            // replace state so that we don't keep the faulty state in history
            GH.RapidBoard.State.replaceState();
        }
    };

    /**
     * Updates the UI after a selection change
     */
    BacklogController.updateUIAfterSelectionChange = function (opts) {
        // update the ui
        BacklogView.updateIssueSelectionState(opts.doScroll);

        // update detail view
        PlanController.updateDetailsView(BacklogSelectionController.getSelectedIssueKey(), opts.openDetailsView);
    };

    /**
     * Fetch the latest data for the epics and update the model and view with it
     * @return a jQuery promise
     */
    BacklogController.updateEpicData = function () {
        if (!BacklogController.visible) {
            return new $.Deferred().resolve();
        }

        var params = {
            rapidViewId: BacklogController.rapidViewData.id
        };

        return GH.Ajax.get({
            url: '/xboard/plan/backlog/epics.json',
            data: params
        }, 'epics').done(function (epicData) {
            if (!BacklogController.visible) {
                return;
            }
            GH.log('loaded data for epics', GH.Logger.Contexts.ajax);
            BacklogController.setEpicData(epicData);
        });
    };

    /**
     * Handles loaded epic data
     */
    BacklogController.setEpicData = function (epicData) {
        // update the model
        GH.EpicController.setEpicData(epicData);

        // update epics
        GH.EpicView.updateEpics();
    };

    /**
     * Fetch the latest data for the versions and update the model and view with it
     * @return a jQuery promise
     */
    BacklogController.updateVersionData = function () {
        if (!BacklogController.visible) {
            return new $.Deferred().resolve();
        }

        var params = {
            rapidViewId: BacklogController.rapidViewData.id,
            activeQuickFilters: PlanControls.quickfilters.getActiveQuickFilters()
        };

        return GH.Ajax.get({
            url: '/xboard/plan/backlog/versions.json',
            data: params
        }, 'versions').done(function (data) {
            if (!BacklogController.visible) {
                return;
            }
            GH.log('loaded data for versions', GH.Logger.Contexts.ajax);
            BacklogController.setVersionData(data.versionData);
        });
    };

    /**
     * Handles loaded version data
     * @param versionData
     */
    BacklogController.setVersionData = function (versionData) {
        // update version data means keeping projects, but that's ok since they can't change without leaving the board
        GH.VersionController.setVersions(versionData);

        // validate the selection
        GH.VersionController.validateSelectedVersion();

        // update versions
        GH.VersionView.updateView();
    };

    /**
     * Moves the issues between two rankables, and possibly a new sprint
     *
     * @param sprintId the sprint to move issues to. null for the backlog
     * @return is the order of issues in the sprint container now in an inconsistent state, requiring a re-order?
     */
    BacklogController.moveIssues = function (issueKeys, sprintId, prevRankableId, nextRankableId) {
        // update the data structure
        var changedModels = BacklogModel.moveIssuesNew(issueKeys, sprintId, prevRankableId, nextRankableId);

        // redraw the view
        BacklogView.redrawChangedModels(changedModels);

        return BacklogModel.hasAnySubtasks(issueKeys);
    };

    /**
     * Reorders the issues in the given sprint container with the given order.
     * Keys that are not present in the sprint container will be ignored.
     *
     * @param issueKeys the keys of the issues in the order that they should be.
     * @param sprintId the container to reorder, null for the backlog
     */
    BacklogController.reorderIssuesInSprint = function (issueKeys, sprintId) {
        var changedModels = BacklogModel.reorderIssuesInModel(issueKeys, sprintId);
        BacklogView.redrawChangedModels(changedModels);
    };

    /**
     * Reverts all statistics
     * This method is called form drag and drop code when the user cancels an update
     */
    BacklogController.revertStatistics = function () {
        SprintView.updateSprintStatistics();
        SprintBacklogView.updateBacklogHeader();
    };

    /**
     * Called when a sprint has been started
     */
    BacklogController.handleSprintStarted = function () {
        BacklogSelectionController.clearSelection();
        // since a sprint has started, put user onto Work mode
        GH.RapidBoard.ViewController.setMode('work');
        GH.RapidBoard.State.pushState();
    };

    BacklogController.handleIssueForEpicCreated = function (event, data) {
        if (!BacklogController.visible) {
            return;
        }

        if (_.isUndefined(BacklogController.handleIssueForEpicCreated.deferred)) {
            BacklogController.handleIssueForEpicCreated.deferred = new $.Deferred();

            // Ajax requests on Firefox fail if issued through an (escape) key event, IF the key event propagation is not stopped.
            // This event could come from quick create (create an issue with "create another" selected, then hit escape)
            // Workaround: do the actual action after a timeout, which will happen outside the keyboard event.
            setTimeout(function () {
                BacklogController.loadData().then(function () {
                    BacklogController.handleIssueForEpicCreated.deferred.resolve();
                    BacklogController.handleIssueForEpicCreated.deferred = undefined;
                }, function () {
                    BacklogController.handleIssueForEpicCreated.deferred.reject();
                });
            }, 0);
        }
        return BacklogController.handleIssueForEpicCreated.deferred;
    };

    BacklogController.registerIssueCreatedCallback = function (id, callback) {
        BacklogController.issueCreatedCallbacks[id] = callback;
    };
    BacklogController.unregisterIssueCreatedCallback = function (id) {
        delete BacklogController.issueCreatedCallbacks[id];
    };

    BacklogController.handleIssueCreated = function (event, data) {
        if (!BacklogController.visible) {
            return;
        }

        // Ajax requests on Firefox fail if issued through an (escape) key event, IF the key event propagation is not stopped.
        // This event could come from quick create (create an issue with "create another" selected, then hit escape)
        // Workaround: do the actual action after a timeout, which will happen outside the keyboard event.

        // copy callbacks, because this might lead to race conditions
        var callbacks = _.clone(BacklogController.issueCreatedCallbacks);

        setTimeout(function () {
            BacklogController.handleIssueCreatedImpl(data, callbacks);
        }, 0);
    };

    BacklogController.handleIssueCreatedImpl = function (data, callbacks) {
        if (data.isSubtask) {
            BacklogController.subtasksCreated(data);
            return;
        }

        function splitIssuesByType(issues) {
            var flatIssues = _.map(issues, function (issue) {
                return {
                    issueKey: issue.key,
                    issueId: issue.id,
                    issueType: issue.fields.issuetype.id
                };
            });

            if (!PlanController.isEpicsPanelEnabled()) {
                return {
                    issues: flatIssues,
                    epics: []
                };
            }

            var epicTypeId = GH.EpicConfig.getEpicIssueTypeId();
            return _.reduce(flatIssues, function (acc, issue) {
                if (issue.issueType === epicTypeId) {
                    acc.epics.push(issue);
                } else {
                    acc.issues.push(issue);
                }
                return acc;
            }, { issues: [], epics: [] });
        }

        // In case of issue/epic creation we don't really have the information from the QuickCreate about what kind of
        // issues were created. So we fire up additional request to get the issue types.
        return BacklogController.getIssueDetails(data.issues, 'issuetype').done(function (result) {
            var issuesByTypes = splitIssuesByType(result.issues);
            BacklogController.issuesAndEpicsCreated(issuesByTypes.issues, issuesByTypes.epics, callbacks);
        });
    };

    BacklogController.getIssueDetails = function (issues, fields) {
        var issueKeys = _.pluck(issues, 'issueKey');
        return BacklogController.getIssueDetailsByKeys(issueKeys, fields);
    };

    BacklogController.getIssueDetailsByKeys = function (issueKeys, fields) {
        var keys = issueKeys.join(',');
        var jql = 'issueKey in (' + keys + ')';
        return GH.Ajax.get({
            bareUrl: GH.Ajax.buildBareRestUrl('/rest/api/2/search'),
            data: { jql: jql, fields: fields }
        }, 'getissues');
    };

    BacklogController.issuesAndEpicsCreated = function (issues, epics, callbacks) {
        if (_.isEmpty(issues) && _.isEmpty(epics)) {
            return;
        }

        // only epics got created and they are in panel state
        if (_.isEmpty(issues) && PlanController.isEpicsPanelEnabled()) {
            return BacklogController.reloadAndShowCreatedEpics(epics);
        }

        // Execute all callbacks
        var prepResults = _.map(callbacks, function (callback) {
            if (_.isFunction(callback)) {
                return callback(issues, epics);
            }

            return false;
        });

        // Filter those that are not immediately done
        var deferreds = _.filter(prepResults, function (def) {
            return def && (def instanceof $.Deferred || _.isFunction(def.promise));
        });

        // Wrap deferreds so they always succeed
        var successDeferreds = _.map(deferreds, function (def) {
            var newDef = $.Deferred();
            if (def.always) {
                def.always(newDef.resolve);
            } else {
                def.then(newDef.resolve, newDef.resolve);
            }
            return newDef;
        });

        // Now we wait for all them to finish and reload the view
        return $.when.apply($, successDeferreds).then(function () {
            BacklogController.reloadAndSelectCreatedIssues(issues);
        });
    };

    BacklogController.subtasksCreated = function (data) {
        // in case of subtasks we show a simple created message and switch over to the subtasks tab
        GH.DetailsViewScrollTracker.setSelectedTab(GH.DetailsView.TAB_SUBTASKS);
        var loadDataPromise;
        if (EditableDetailsViewReloadReason) {
            loadDataPromise = BacklogController.loadData(EditableDetailsViewReloadReason.SUBTASKS_CHANGED);
        } else {
            loadDataPromise = BacklogController.loadData();
        }

        loadDataPromise.done(function () {

            BacklogController.selectCreatedIssues(data.issues);
            BacklogController.expandParents(data.issues);

            if (shouldShowInvisibleSubtasksCreatedMessage(data)) {
                GH.RapidBoard.QuickCreate.showSubtaskCreatedMessage(data);
            }
        });
    };

    function shouldShowInvisibleSubtasksCreatedMessage(data) {
        if (GH.RapidBoard.State.isKanbanBoard()) {
            var issueKeys = _.map(data.issues, function (issue) {
                return issue.issueKey;
            });
            return BacklogModel.hasAnyIssueInvisible(issueKeys);
        }
        return true;
    }

    BacklogController.reloadAndSelectCreatedIssues = function (issues) {
        return BacklogController.loadData().done(function () {
            var issueSelected = BacklogController.selectCreatedIssues(issues);

            if (!issueSelected) {
                GH.RapidBoard.QuickCreate.showCreatedIssuesMessage({ issues: issues });
            }
        });
    };

    BacklogController.selectCreatedIssues = function (issues) {
        var lastIssue = _.last(issues);
        var selectIssue = BacklogModel.isIssueVisible(lastIssue.issueKey);
        if (selectIssue) {
            BacklogSelectionController.selectIssue(lastIssue.issueKey); // select last issue we created
        }

        return !!selectIssue;
    };

    /**
     * Expand parent of issues
     * @param issues
     */
    BacklogController.expandParents = function (issues) {
        issues.forEach(function (issue) {
            SubtasksExpandingController.toggleExpandStateForKey(issue.issueKey, true);
        });
    };

    BacklogController.reloadAndShowCreatedEpics = function (epics) {
        BacklogController.updateEpicData().done(function () {
            BacklogController._showCreatedEpicsMessage(epics);

            GH.EpicView.renderNewEpics(epics);
        });
    };

    BacklogController._showCreatedEpicsMessage = function (epics) {
        var lastIssue = _.last(epics);

        var epicList = GH.EpicController.getEpicModel().getEpicList();
        var isEpicVisible = epicList.isIssueVisible(lastIssue.issueKey);

        GH.RapidBoard.QuickCreate.showCreatedEpicsMessage({ issues: epics }, isEpicVisible);
        if (isEpicVisible) {
            GH.EpicView.scrollToBottom();
        }
    };

    BacklogController.handleIssueUpdated = function (event, data) {
        if (!BacklogController.visible) {
            return;
        }
        GH.log(event.type + " from source " + data.source + " handled", "BacklogController");

        // did we edit an issue in an active sprint?
        // get the list for this issue
        var issueId = parseInt(data.issueId, 10);

        var editedIssueModel = BacklogModel.findModelWithIssue(issueId);
        // When we are in Plan mode, subtasks will not represent in the model
        var isSubtaskEdited = !editedIssueModel;
        var needsDetailViewReload = false;
        var needsEpicsReload = false;
        var needsVersionsReload = false;
        var reloadAllData = false;
        var issueData;

        // quick edit update?
        if (data.source === 'quickEdit') {
            // quick edit triggered update

            // detail view is outdated
            needsDetailViewReload = true;

            // we don't know what fields got edited
            needsEpicsReload = true;
            needsVersionsReload = true;

            // if we haven't got a model containing the issue we most likely edited a subtask. In case where a tracking statistic is enabled
            // we'll have to reload all data because the tracking statistic is accumulated into the parent.
            // the tracking statistic is currently displayed in the future sprint footer when different from the OE
            if (!editedIssueModel && BacklogModel.trackingStatistic.isEnabled) {
                reloadAllData = true;
            }
        }

        // detail view update
        else {
                // assume source is detail view inline edit
                // only update epic when the estimation statistic or epic is changed
                if (!_.isUndefined(data.fieldId)) {
                    needsEpicsReload = data.fieldId === BacklogController.rapidViewConfig.estimationStatistic.fieldId || data.fieldId === GH.EpicConfig.getEpicLinkFieldId();

                    // only update versions when a version field is changed
                    needsVersionsReload = data.fieldId === 'fixVersions' || data.fieldId === 'versions' || data.fieldId === BacklogController.rapidViewConfig.estimationStatistic.fieldId;
                }
            }

        // if the sprint field has changed, we reload all data
        if (data.fieldChanges) {
            var sprintFieldChanges = data.fieldChanges[SprintConfig.getSprintFieldId()];
            if (sprintFieldChanges && sprintFieldChanges.original != sprintFieldChanges.updated) {
                reloadAllData = true;
            }
        }

        // reload all data if necessary
        if (reloadAllData) {
            if (EditableDetailsViewReloadReason && isSubtaskEdited) {
                BacklogController.loadData(EditableDetailsViewReloadReason.SUBTASKS_CHANGED);
            } else {
                BacklogController.loadData();
            }
        }

        // otherwise reload individual pieces
        else {
                // versions
                if (needsVersionsReload) {
                    BacklogController.updateVersionData().always(BacklogController.reloadSingleIssue.bind(null, data.issueId));
                } else {
                    // reload the issue
                    BacklogController.reloadSingleIssue(data.issueId);
                }

                if (needsDetailViewReload) {
                    if (EditableDetailsViewReloadReason && isSubtaskEdited) {
                        PlanController.reloadDetailView(EditableDetailsViewReloadReason.SUBTASKS_CHANGED);
                    } else {
                        PlanController.reloadDetailView();
                    }
                }

                // epics
                if (needsEpicsReload) {
                    BacklogController.updateEpicData();
                }
            }
    };

    /**
     * Handles the removed from sprint event fired by the remove from sprint issue action.
     */
    BacklogController.handleIssuesRemovedFromSprint = function (event, data) {
        if (!BacklogController.visible) {
            return;
        }

        if (data.issueKeys.length === 1) {
            //no need to update the UI here, it will be done after when the board will be reloaded
            BacklogSelectionController.selectIssue(data.issueKeys[0]);
            BacklogSelectionController.selectionManager._addSelectedIssueKeys(data.issueKeys);
        }

        // reload the backlog
        BacklogController.loadData();
    };

    /**
     * Updates a single issue by loading the data from the server
     *
     * The resulting update will only update the backlog/sprintquery, and not affect epics, versions and detail view.
     */
    BacklogController.reloadSingleIssue = function (issueId) {
        var params = {
            rapidViewId: BacklogController.rapidViewData.id,
            issueId: issueId,
            activeQuickFilters: PlanControls.quickfilters.getActiveQuickFilters()
        };

        GH.Ajax.get({
            url: '/xboard/plan/backlog/issue.json',
            data: params
        }, 'updateSingleIssue.' + issueId).done(function (issue) {
            GH.log('loaded data for single issue with id ' + issueId, GH.Logger.Contexts.ajax);
            BacklogController.updateIssues(issue);
        });
    };

    /**
     * Updates backlog issues with new data and updates the affected sprints/backlog views.
     *
     * This method also handles the case where the updated data signals an issue change from story to epic. In this case the
     * issue is removed from the backlog and the epics panel reloaded
     */
    BacklogController.updateIssues = function (issues) {
        if (!_.isArray(issues)) {
            issues = [issues];
        }

        // update all issues but retain which lists changed
        var changedModels = [];
        var issueTypeChangeToEpic = false;

        // update each issue in the corresponding model
        _.each(issues, function (issue) {
            // find the model the issue is currently part of
            var model = BacklogModel.findModelWithIssue(issue.key);
            // can't do much if we don't know about the issue
            if (!model) {
                return;
            }

            if (!issue.parentKey) {
                // Find the models that has a fake parent stub for the issue. We need to update it as well
                var modelsWithFakes = BacklogModel.findModelsWithFakeParent(issue.key);
                [].push.apply(changedModels, modelsWithFakes);
            }

            // When epics panel is visible, we don't display epics with other issues in the backlog.
            // Hence, we need to handle the case where an issue got changed into an epic.
            if (PlanController.isEpicsPanelEnabled() && issue.typeId === GH.EpicConfig.getEpicIssueTypeId()) {
                model.getIssueList().removeIssues(issue.key);
                issueTypeChangeToEpic = true;
            } else {
                // update the issue
                model.getIssueList().updateIssue(issue);
            }
            changedModels.push(model);
            BacklogModel.afterIssueUpdate(issue);
        });

        // re-apply current search filters so that updated issues are re-considered as hidden or visible
        BacklogModel.updateFiltering();

        // handle the issue type change
        if (issueTypeChangeToEpic) {
            // load the epics data again
            BacklogController.updateEpicData();

            // validate the current issue selection and update the detail view (with a likely new selection)
            BacklogSelectionController.validateCurrentSelection();
            var selectedIssueKey = BacklogSelectionController.getSelectedIssueKey();
            GH.DetailsView.setSelectedIssueKey(selectedIssueKey);
            PlanController.reloadDetailView();
        }

        // we don't want to redraw models multiple times
        changedModels = _.uniq(changedModels);

        // redraw the changed models
        BacklogView.redrawChangedModels(changedModels);
    };

    /**
     * Removes one or several issues from a model, then redraws the ui
     * This method is called from optimistic drag and drop updates
     */
    BacklogController.removeIssues = function (model, issueKeys) {
        model.getIssueList().removeIssues(issueKeys);
        BacklogView.redrawChangedModel(model);
    };

    //
    // Filtering
    //

    /**
     * Check whether filters are currently active on the Backlog
     */
    BacklogController.isFiltersActive = function () {
        return !_.isEmpty(PlanControls.quickfilters.getActiveQuickFilters());
    };

    BacklogController.executeSearch = function (pattern) {
        GH.Logger.timeStart("BacklogController.executeSearch regex");

        if (_.isNull(pattern)) {
            BacklogController.lastExecutedSearchFilter = null;
        } else {
            // this breaks encapsulation somewhat in that we know that the builder of the regex
            // has used whitespace as the splitter of the input to build the regex
            // but...we need something.  We need an out of bounds string
            var oobStr = ' ';

            // construct a function to be applied to issues to determine their visibility
            BacklogController.lastExecutedSearchFilter = function (issue) {
                issue = issue || {};
                return pattern.test(issue.key + oobStr + issue.summary + oobStr + issue.typeName + oobStr + issue.assignee + oobStr + issue.assigneeName);
            };
        }

        // set new filter
        PlanIssueListFiltering.setInstantFilter(BacklogController.lastExecutedSearchFilter);

        // apply filter to model
        var changed = BacklogModel.updateFiltering();

        GH.Logger.timeStop("BacklogController.executeSearch regex");
        GH.Logger.timeStart("BacklogController.executeSearch ui update");

        if (changed) {
            // update the hidden issues as well as the marker
            BacklogView.updateHiddenIssues();
        }

        GH.Logger.timeStop("BacklogController.executeSearch ui update");

        return false;
    };

    //
    // Clears ALL the things (aka filters)
    //
    BacklogController.clearFilters = function () {

        // first clear the quick filters so we know whether we have to reload the data
        var clearQuickFilters = PlanControls.quickfilters.clearFilters();

        // remove the other filters
        PlanControls.searchFilter.clearSearchBox();
        GH.VersionController.setFilteredVersionId(null);
        GH.EpicController.setFilteredEpicKey(null);
        GH.EpicController.clearVersionFiltering();

        if (clearQuickFilters) {
            // clear
            PlanIssueListFiltering.clearAllFilters();

            // then reload
            BacklogController.loadData().done(function () {
                GH.RapidBoard.State.pushState();
            });
        } else {
            // clear filters
            var changed = PlanIssueListFiltering.clearAllFilters();

            // update UI
            if (changed) {
                // epics and versions
                GH.EpicView.deselectAllEpics();
                GH.VersionView.deselectAllVersions();

                // update model (hidden issues)
                var issuesChanged = BacklogModel.updateFiltering();

                // update issues if necessary
                if (issuesChanged) {
                    BacklogView.updateHiddenIssues();
                }

                GH.RapidBoard.State.pushState();
            }
        }
    };

    /**
     * When an epic is selected, we filter the visible backlog and open sprint issues to be only the ones that are associated
     * to the epic. We must also ensure that all statistics are recalculated.
     *
     * @param skipRendering set to true if we don't need to render any changes
     */
    BacklogController.updateEpicFiltering = function (skipRendering) {
        // update the filtering
        var epicKey = GH.EpicController.getFilteredEpicKey();
        PlanIssueListFiltering.setEpicFilter(epicKey);

        // apply to all list models
        var changed = BacklogModel.updateFiltering();

        // update UI if something changed
        if (changed && !skipRendering) {
            BacklogView.updateHiddenIssues();
        }
    };

    /**
     * Update the version filtering according to the currently selected version
     * @param skipRendering true to not render the view
     */
    BacklogController.updateVersionFiltering = function (skipRendering) {
        // update the filtering
        var versionId = GH.VersionController.getFilteredVersionId();
        PlanIssueListFiltering.setVersionFilter(versionId);

        // apply to all list models
        var changed = BacklogModel.updateFiltering();

        // update the UI if requested
        if (changed && !skipRendering) {
            BacklogView.updateHiddenIssues();
        }

        // update the epic filtering accordingly
        // TODO: this should be inside above's if statement!
        GH.EpicController.applyVersionFiltering();
    };

    // Render data calculation

    BacklogController.calculateIssueRenderData = function () {
        var selectedIssues = BacklogSelectionController.getSelectedIssueKeys();
        var selectedIssueKeys = {};
        _.each(selectedIssues, function (issueKey) {
            selectedIssueKeys[issueKey] = true;
        });
        var mainSelectedIssue = BacklogSelectionController.getSelectedIssueKey();
        var hiddenIssues = BacklogModel.getHiddenBySearchIssues();
        return {
            selectedIssueKeys: selectedIssueKeys,
            mainSelectedIssue: mainSelectedIssue,
            hiddenIssues: hiddenIssues
        };
    };

    /**
     * Extracts the list of issues from all sprints of the current plan view. Extracted issues are the ones that
     * are not hidden .
     *
     * @return {{sprintId: int, sprintName: string, issues: Array}[]}
     */
    BacklogController.extractPrintableIssuesFromAllSprints = function () {
        var sprintIssues = [];

        _.each(BacklogModel.getSprintModels(), function (sprintModel) {
            var sprintData = sprintModel.getSprintData();
            var issues = extractVisibleIssues(sprintModel.getIssueList().getAllIssues(), sprintModel.getIssueList().getVisibleRankables(), BacklogModel.getHiddenBySearchIssues());

            if (!_.isEmpty(issues)) {
                sprintIssues.push({
                    sprintId: sprintData.id,
                    sprintName: sprintData.name,
                    issues: issues
                });
            }
        });
        return sprintIssues;
    };

    /**
     * Returns issues that are not hidden (not exist in {hiddenIssues} array) from {allIssues} array and sort them
     * in the same order as issue keys in the {orderedIssuesKey} array.
     *
     * @param {Object[]} allIssues array of both visible issues and hidden issues
     * @param {string[]} orderedIssuesKey array of issues keys in the expected displayed order.
     * @param hiddenIssues array of hidden issues. if (hiddenIssues[issueKey}) then that issue is hidden.
     *
     * @return {Object[]} array of visible issues to print in the expected order.
     */
    function extractVisibleIssues(allIssues, orderedIssuesKey, hiddenIssues) {
        var issues = [];
        // loop through issue key list and add corresponding issue of each key, by this way, the list of
        // returned issues will be in the same order as the issue keys.
        _.each(orderedIssuesKey, function (key) {
            if (!hiddenIssues[key]) {
                issues.push(allIssues[key]);
            }
        });
        return issues;
    }

    /**
     * Extracts the list of issues from the backlog of the current plan view. Extracted issues are the ones that
     * are not hidden.
     *
     * @return {Object[]} an array of issues. Never null
     */
    BacklogController.getPrintableIssuesFromPlanBacklog = function () {
        var backlogIssueList = BacklogModel.getBacklogModel2().getIssueList();
        return extractVisibleIssues(backlogIssueList.getAllIssues(), backlogIssueList.getVisibleRankables(), BacklogModel.getHiddenBySearchIssues());
    };

    /**
     * @returns {*|{}|map} epic issue key -> epic issue
     */
    BacklogController.getEpicMap = function () {
        return GH.EpicController.getEpicModel().getEpicList().getAllIssues();
    };

    return BacklogController;
});