AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common", "com.pyxis.greenhopper.jira:gh-rapid"], function () {
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogModel2 = require('jira-agile/rapid/ui/plan/backlog-model2');
    var _ = require('underscore');

    module("BacklogModel tests");

    test("module 'jira-agile/rapid/ui/plan/backlog-model' exists", function () {
        ok(BacklogModel);
        ok(GH.BacklogModel);
    });

    test("module 'jira-agile/rapid/ui/plan/backlog-model2' exists", function () {
        ok(BacklogModel2);
        ok(GH.BacklogModel2);
    });

    test("clearing filters", function () {
        GH.PlanIssueListFiltering.clearAllFilters();
        deepEqual(GH.PlanIssueListFiltering.allFilters, {});
    });

    module("BacklogModel.getAllIssuesWithMissingParents");

    function mockIssue(key, statusId, options) {
        return _.extend({}, {
            statusId: statusId,
            key: key
        }, options);
    }

    function mockColumn(statusId) {
        return {
            statusIds: [statusId],
            name: statusId
        };
    }

    function createCompareData() {
        var returnData = {};

        for (var i = 0; i < arguments.length; i++) {
            returnData[arguments[i].key] = arguments[i];
        }

        return returnData;
    }

    test("should return concatenated values from all models and missingParentsByKey", function () {
        var missingIssue = mockIssue("missing issue", 3, {
            additionalOption: true
        });

        var backlogIssue = mockIssue("backlog issue", 1, {
            yetAnotherAdditionalOption: false
        });
        var selectedForDevIssue = mockIssue("selected for dev issue", 2, {
            yesItIsNewerAdditionalOption: [1]
        });
        var selectedForDevIssue2 = mockIssue("selected for dev issue 2", 2, {
            newestAdditionalOption: [1]
        });

        BacklogModel.setData({
            missingParents: [missingIssue],

            backlogColumn: mockColumn(1),
            selectedForDevelopmentColumn: mockColumn(2),

            issues: [backlogIssue, selectedForDevIssue, selectedForDevIssue2]
        });

        deepEqual(BacklogModel.getAllIssuesWithMissingParents(), createCompareData(missingIssue, backlogIssue, selectedForDevIssue, selectedForDevIssue2));
    });

    test("missingParentsByKey should override values from models", function () {
        var missingIssue = mockIssue("overriden issue", 3, {
            thisOptionShouldBeVisible: "yes, I'm visible"
        });

        var overridenIssue = mockIssue("overriden issue", 1, {
            thisOptionShouldNotBeVisible: "I'm overriden, sorry"
        });

        BacklogModel.setData({
            missingParents: [missingIssue],

            backlogColumn: mockColumn(1),
            selectedForDevelopmentColumn: mockColumn(2),

            issues: [overridenIssue]
        });

        deepEqual(BacklogModel.getAllIssuesWithMissingParents(), createCompareData(missingIssue));
    });

    module('hasAnyIssueInvisible', {
        teardown: function teardown() {
            BacklogModel.isIssueVisible.restore();
        }
    });

    test('hasAnyIssueInvisible() should return true when any of the given issue is invisible', function () {
        var invisibleIssueKey = 'TEST-1';
        var visibleIssueKey = 'TEST-2';
        var issueKeys = [invisibleIssueKey, visibleIssueKey];
        sinon.stub(BacklogModel, 'isIssueVisible').withArgs(invisibleIssueKey).returns(false).withArgs(visibleIssueKey).returns(true);
        ok(BacklogModel.hasAnyIssueInvisible(issueKeys) === true);
    });

    test('hasAnyIssueInvisible() should return false when all issues are visible', function () {
        var issueKeys = ['TEST-1', 'TEST-2'];
        sinon.stub(BacklogModel, 'isIssueVisible').withArgs('TEST-1').returns(true).withArgs('TEST-2').returns(true);
        ok(BacklogModel.hasAnyIssueInvisible(issueKeys) === false);
    });

    module("BacklogModel reorderIssuesInModel", {
        setup: function setup() {
            this.issueKeys = ['KEY-1', 'KEY-2'];

            var backlogModels = {};
            backlogModels['backlog'] = new MockBacklogModel('backlog');
            backlogModels[-1] = new MockBacklogModel(-1);
            backlogModels[1] = new MockBacklogModel(1);

            sinon.stub(BacklogModel, 'getAllModels').returns(_.values(backlogModels));

            this.backlogModels = backlogModels;
        },
        teardown: function teardown() {
            BacklogModel.getAllModels.restore();
        }
    });

    function MockBacklogModel(id) {
        var allIssues = {
            'KEY-1': {},
            'KEY-2': {}
        };

        var issueList = {
            setOrder: sinon.stub(),
            getAllIssues: function getAllIssues() {
                return allIssues;
            }
        };

        this.getId = function () {
            return id;
        };
        this.getIssueList = function () {
            return issueList;
        };
    }

    test("test Backlog", function () {
        var sprintId = undefined;
        var changedModels = BacklogModel.reorderIssuesInModel(this.issueKeys, sprintId);

        equal(changedModels.length, 1);
        equal(changedModels[0], this.backlogModels['backlog']);

        this.backlogModels['backlog'].getIssueList().setOrder.calledWith(this.issueKeys);
    });

    test("test SelectedForDevelopment", function () {
        var sprintId = -1;
        var changedModels = BacklogModel.reorderIssuesInModel(this.issueKeys, sprintId);

        equal(changedModels.length, 1);
        equal(changedModels[0], this.backlogModels[-1]);

        this.backlogModels[-1].getIssueList().setOrder.calledWith(this.issueKeys);
    });

    test("test Sprint 1", function () {
        var sprintId = 1;
        var changedModels = BacklogModel.reorderIssuesInModel(this.issueKeys, sprintId);

        equal(changedModels.length, 1);
        equal(changedModels[0], this.backlogModels[1]);

        this.backlogModels[1].getIssueList().setOrder.calledWith(this.issueKeys);
    });

    test("test invalid model", function () {
        var sprintId = 50;
        var changedModels = BacklogModel.reorderIssuesInModel(this.issueKeys, sprintId);

        equal(changedModels.length, 0);

        _.each(this.backlogModels, function (model) {
            ok(model.getIssueList().setOrder.notCalled);
        });
    });

    test("test ignores issues not already in model", function () {
        var sprintId = 1;
        var changedModels = BacklogModel.reorderIssuesInModel(["KEY-500", "KEY-1", "KEY-3", "KEY-2"], sprintId);

        equal(changedModels[0], this.backlogModels[1]);

        this.backlogModels[1].getIssueList().setOrder.calledWith(["KEY-1", "KEY-2"]);
    });

    test("test does not include issues that should no longer be in the model", function () {
        var sprintId = 1;
        var changedModels = BacklogModel.reorderIssuesInModel(["KEY-2"], sprintId);

        equal(changedModels[0], this.backlogModels[1]);

        this.backlogModels[1].getIssueList().setOrder.calledWith(["KEY-2"]);
    });

    module("BacklogModel hasAnySubtasks", {
        setup: function setup() {
            var issues = {
                'KEY-1': new MockIssue(),
                'KEY-2': new MockSubtask(),
                'KEY-3': new MockIssue(),
                'KEY-4': new MockIssue()
            };
            sinon.stub(BacklogModel, 'getIssueData', function (key) {
                return issues[key];
            });
        },
        teardown: function teardown() {
            BacklogModel.getIssueData.restore();
        }
    });

    function MockSubtask() {
        this.parentId = 1000;
        this.parentKey = 'PARENT-KEY';
    }

    function MockIssue() {}

    test("hasAnySubtasks when selection includes subtask should be true", function () {
        var hasAny = BacklogModel.hasAnySubtasks(["KEY-1", "KEY-2", "KEY-3"]);

        ok(hasAny);
    });

    test("hasAnySubtasks when selection does not include subtask should be false", function () {
        var hasAny = BacklogModel.hasAnySubtasks(["KEY-1", "KEY-3", "KEY-4"]);

        ok(!hasAny);
    });

    test("hasAnySubtasks for empty selection should be false", function () {
        var hasAny = BacklogModel.hasAnySubtasks([]);

        ok(!hasAny);
    });

    module('issue parents', {
        setup: function setup() {
            var backlogStatus = 1;
            var devStatus = 2;
            this.backlogColumn = mockColumn(backlogStatus);
            this.devColumn = mockColumn(devStatus);
            this.BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
            this.parent = mockIssue('PROJ-1', backlogStatus);
            this.subTask = mockIssue('PROJ-2', backlogStatus);
            this.subTask.parentKey = this.parent.key;
            this.withoutParent = mockIssue('PROJ-3', backlogStatus);

            this.getAllIssues = function () {
                return this.BacklogModel.getAllIssueListsNew().map(function (x) {
                    return x.getAllIssues();
                }).reduce(_.defaults);
            };
        }
    });

    test('every sub-task should have a parent', function (assert) {
        BacklogModel.setData({
            issues: [this.parent, this.subTask, this.withoutParent],
            backlogColumn: this.backlogColumn,
            selectedForDevelopmentColumn: this.devColumn
        });
        var issues = this.getAllIssues();
        assert.deepEqual(issues['PROJ-2'].parent, this.parent);
    });

    test('every sub-task should have a parent when the parent is hidden', function (assert) {
        BacklogModel.setData({
            issues: [this.withoutParent, this.subTask],
            missingParents: [this.parent],
            backlogColumn: this.backlogColumn,
            selectedForDevelopmentColumn: this.devColumn
        });
        var issues = this.getAllIssues();
        assert.deepEqual(issues['PROJ-2'].parent, this.parent);
    });

    test('sub-task\'s parent is set after update', function (assert) {
        BacklogModel.setData({
            issues: [this.withoutParent, this.subTask],
            missingParents: [this.parent],
            backlogColumn: this.backlogColumn,
            selectedForDevelopmentColumn: this.devColumn
        });

        var issues = this.BacklogModel.getAllIssuesWithMissingParents();
        var updatedSubTask = _.extend({}, issues['PROJ-2'], { parent: undefined });
        this.BacklogModel.findModelWithIssue('PROJ-2').getIssueList().updateIssue(updatedSubTask);
        this.BacklogModel.afterIssueUpdate(updatedSubTask);

        assert.deepEqual(this.BacklogModel.getAllIssuesWithMissingParents()['PROJ-2'].parent, this.parent);
    });

    test('sub-tasks are updated after when parent is updated', function (assert) {
        BacklogModel.setData({
            issues: [this.withoutParent, this.subTask],
            missingParents: [this.parent],
            backlogColumn: this.backlogColumn,
            selectedForDevelopmentColumn: this.devColumn
        });

        var issues = this.BacklogModel.getAllIssuesWithMissingParents();
        var updatedParent = _.extend({}, issues['PROJ-1'], { summary: 'new summary' });
        BacklogModel.afterIssueUpdate(updatedParent);
        assert.deepEqual(this.BacklogModel.getAllIssuesWithMissingParents()['PROJ-2'].parent, updatedParent);
    });
});