/* globals
 * GH.IssueSelectionManager, GH.MouseUtils, GH.RapidBoard
 */

/**
 * Backlog issue support
 * @module jira-agile/rapid/ui/plan/backlog-selection-controller
 * @requires module:jquery
 * @requires module:jira-agile/rapid/ui/plan/backlog-model
 * @requires module:jira-agile/rapid/ui/plan/BacklogView
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-issue-list-view
 * @requires module:jira-agile/rapid/ui/plan/plan-context-menu-controller
 */
define('jira-agile/rapid/ui/plan/backlog-selection-controller', ['require'], function(require) {
    'use strict';

    var $ = require('jquery');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var PlanIssueListView = require('jira-agile/rapid/ui/plan/plan-issue-list-view');
    var PlanContextMenuController = require('jira-agile/rapid/ui/plan/plan-context-menu-controller');
    var SubtasksExpandingController = require('jira-agile/rapid/ui/plan/subtasks-expanding-controller');
    var BacklogController;
    var BacklogView;

    // Resolve circular dependencies
    GlobalEvents.on('pre-initialization', function() {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
        BacklogView = require('jira-agile/rapid/ui/plan/BacklogView');
    });

    var BacklogSelectionController = {};

    BacklogSelectionController.selectionDataModel = {};

    /**
     * Selection manager object
     */
    BacklogSelectionController.selectionManager = undefined;

    /**
     * Called upon page initialization
     */
    BacklogSelectionController.init = function () {
        // create a proxy model object
        BacklogSelectionController.selectionDataModel = {
            isIssueValid: BacklogModel.isIssueSelectable,
            getIssueIndex: BacklogModel.getIssueIndex,
            canAddToSelection: BacklogModel.canAddToSelection,
            getIssueRange: BacklogModel.getIssueRange
        };

        // create the selection manager used by this object
        BacklogSelectionController.selectionManager = new GH.IssueSelectionManager(BacklogSelectionController.selectionDataModel, 'backlog.selectedIssues');

        // issue selection
        $(document).on('click', BacklogView.containerSelector + ' .js-issue', BacklogSelectionController.handleIssueClick);

        // issue context menu
        $(document).on('contextmenu', BacklogView.containerSelector + ' .js-issue', BacklogSelectionController.handleIssueContextMenu);

        // subtasks expander
        $(document).on('click', BacklogView.containerSelector + ' .js-issue .ghx-card-expander', BacklogSelectionController.handleParentExpanderClick);
    };

    BacklogSelectionController.handleIssueContextMenu = function (event) {
        // When right clicking a non selected issue, it should become selected before showing the context menu
        var issueKey = PlanIssueListView.getIssueKey($(this));
        var inSelection = BacklogSelectionController.isInSelection(issueKey);
        if (!inSelection) {
            // pass through the context menu action as if it was a normal click.
            BacklogSelectionController.selectionManager.handleIssueClick(GH.MouseUtils.getMetaKeys(event), issueKey);
            BacklogSelectionController.updateUIAndState({openDetailsView: false, doScroll: true});
        }
        PlanContextMenuController.showContextMenuAt(event);
    };

    /**
     * Handles issue click events
     */
    BacklogSelectionController.handleIssueClick = function (event) {
        // fetch the key for the given issue
        var issueKey = PlanIssueListView.getIssueKey($(this));
        var isTargetIssueKey = $(event.target).hasClass('js-key-link');
        var shouldOpenInNewTab = !GH.MouseUtils.openInSameTab(event);

        if (isTargetIssueKey && shouldOpenInNewTab) {
            return true;
        }

        //prevent any default browser click actions on the cards
        event.preventDefault();

        // handle issue selection click
        BacklogSelectionController.handleIssueSelectionEvent(event, issueKey, true);
    };

    BacklogSelectionController.validateCurrentSelection = function validateCurrentSelection() {
        // check whether the current selection is still valid
        var outcome = BacklogSelectionController.selectionManager.validateCurrentSelection();

        // update if something in the selection has changed
        if (outcome) {
            // the current url state is most probably invalid, so replace it with the new selection
            GH.RapidBoard.State.replaceState();

            BacklogSelectionController.deferredPushState();
        }

        return outcome;
    };

    /**
     * Handles an issue selection event
     */
    BacklogSelectionController.handleIssueSelectionEvent = function (event, issueKey, openDetailView) {
        var metaKeys = GH.MouseUtils.getMetaKeys(event);
        var outcome = BacklogSelectionController.selectionManager.handleIssueClick(metaKeys, issueKey);
        var isMultiSelect = GH.IssueSelectionManager.getMultiSelectType(metaKeys);

        if (isMultiSelect) {
            openDetailView = false;
        }

        if (outcome) {
            BacklogSelectionController.updateUIAndState({openDetailsView: openDetailView, doScroll: false});
        }
    };

    BacklogSelectionController.handleParentExpanderClick = function (event) {
        //prevent issue selection event
        event.stopPropagation();

        var $issue = PlanIssueListView.getClosestIssueElement($(this));
        var issueKey = PlanIssueListView.getIssueKey($issue);
        var expanded = SubtasksExpandingController.toggleExpandState($issue);

        //select issue after collapse
        if (!expanded) {
            BacklogSelectionController.selectIssue(issueKey);
        }

        SubtasksExpandingController.sendAnalytics(expanded);
    };

    BacklogSelectionController.selectIssue = function (issueKey) {
        BacklogSelectionController.selectionManager.selectIssue(issueKey);
        // Always open the detail view
        BacklogSelectionController.updateUIAndState({openDetailsView: false, doScroll: true});
    };

    BacklogSelectionController.selectAllCollapsedSubtasks = function () {
        var collapsedSubtasks = SubtasksExpandingController.getCollapsedSubTasksToSelection(BacklogSelectionController.getSelectedIssueKeys());
        BacklogSelectionController.selectionManager._prependSelectedIssueKeys(collapsedSubtasks);
    };

    BacklogSelectionController.clearSelection = function () {
        BacklogSelectionController.selectIssue(null);
    };

    BacklogSelectionController.selectNextIssue = function () {
        var next = null;
        var selected = BacklogSelectionController.getSelectedIssueKey();
        if (selected) {
            next = BacklogModel.getNextIssueKey(selected);
        } else {
            // fetch the first issue
            next = BacklogModel.getFirstRankableIssueKeyInColumn(null);
        }
        // stop ourselves from running off the top, so that the only time we select null is when there are no issues
        if (!next) {
            next = selected;
        }

        BacklogSelectionController.selectIssue(next);
    };

    BacklogSelectionController.selectPreviousIssue = function () {
        var previous = null;
        var selected = BacklogSelectionController.getSelectedIssueKey();
        if (selected) {
            previous = BacklogModel.getPreviousIssueKey(selected);
        } else {
            // fetch the last issue
            previous = BacklogModel.getLastRankableIssueKeyInColumn(null);
        }
        // stop ourselves from running off the top, so that the only time we select null is when there are no issues
        if (!previous) {
            previous = selected;
        }

        BacklogSelectionController.selectIssue(previous);
    };

    /**
     * Updates the ui and the url state. Commonly called when the issue selection changed
     */
    BacklogSelectionController.updateUIAndState = function (opts) {
        // select all collapsed subtasks
        BacklogSelectionController.selectAllCollapsedSubtasks();

        // update the ui
        BacklogController.updateUIAfterSelectionChange(opts);

        // update the url state
        BacklogSelectionController.deferredPushState();
    };

    // Stall issue selection updates to speed up j/k operation with pressed down key
    BacklogSelectionController.pushTimeoutHandler = undefined;
    BacklogSelectionController.deferredPushState = function () {
        var stallDuration = 1000;
        var now = new Date();

        // clear previous timeout handler
        if (GH.RapidBoard.UrlState.pushTimeoutHandler) {
            window.clearTimeout(GH.RapidBoard.UrlState.pushTimeoutHandler);
        }

        // register new handler
        GH.RapidBoard.UrlState.pushTimeoutHandler = window.setTimeout(function () {
            GH.RapidBoard.UrlState.deferredPushState(now);
        }, stallDuration);
    };

    /**
     * Ensure that when an event happens to a particular issue, that the issue is selected.
     * @param event
     * @param issueKey
     */
    BacklogSelectionController.ensureIssueSelected = function (event, issueKey) {
        // if the dragged issue is not selected, we want to select it
        if (!BacklogSelectionController.selectionManager.isInSelection(issueKey)) {
            // let the selection controller handle the click if the issue is compatible with the current selection
            if (BacklogModel.canAddToSelection(BacklogSelectionController.selectionManager.getSelectedIssueKeys(), issueKey)) {
                // treat as if this were a normal click event
                BacklogSelectionController.handleIssueSelectionEvent(event, issueKey);
            } else {
                // otherwise force a new selection
                BacklogSelectionController.selectionManager.selectIssue(issueKey);
                var openDetailsView = true;
                // Not open detail view if it is being closed
                if (!GH.RapidBoard.State.isDetailsViewPlanOpened()) {
                    openDetailsView = false;
                }
                BacklogSelectionController.updateUIAndState({openDetailsView: openDetailsView, doScroll: true});
            }
        }
    };

    /**
     * Returns all the currently selected issues.
     */
    BacklogSelectionController.getSelectedIssueKeys = function () {
        return BacklogSelectionController.selectionManager.getSelectedIssueKeys();
    };

    /** Get the main selected issue (this is the one that has been last selected) */
    BacklogSelectionController.getSelectedIssueKey = function () {
        return BacklogSelectionController.selectionManager.getSelectedIssueKey();
    };

    /** Is a given issue in the selection */
    BacklogSelectionController.isInSelection = function (issueKey) {
        return BacklogSelectionController.selectionManager.isInSelection(issueKey);
    };

    /**
     * Called by PlanUrlState to set the selection as defined in the url
     */
    BacklogSelectionController.setIssueFromUrlState = function (issueKey) {
        BacklogSelectionController.selectionManager.selectIssue(issueKey);
    };

    /**
     * Returns all the currently selected issues in order of rank.
     */
    BacklogSelectionController.getSelectedIssuesInOrder = function () {
        // TODO: directly implement this here, instead of pushing it to the selection manager
        return BacklogSelectionController.selectionManager.getSelectedIssuesInOrder();
    };

    /**
     * Returns all the currently selected issues and filtered by visibility in order of rank.
     */
    BacklogSelectionController.getSelectedAndVisibleIssuesInOrder = function () {
        return BacklogSelectionController.filterIssuesThatCanBeRanked(
            BacklogSelectionController.getSelectedIssuesInOrder());
    };

    /**
     * Filter issues that shouldn't be transitioned - e.g. hidden by filters (but not collapsed subtasks)
     * @param {Array} issues list of issue keys  to be filtered
     */
    BacklogSelectionController.filterIssuesThatCanBeRanked = function(issues) {
        var hiddenIssues = BacklogModel.getHiddenBySearchIssues();

        function isIssueVisible(issueData) {
            return !issueData.hidden && !hiddenIssues[issueData.key];
        }

        return issues.filter(function(issueKey) {
            var model = BacklogModel.findModelWithIssue(issueKey);

            if (!model) {
                return false;
            }

            var issueData = model.getIssueList().getIssueData(issueKey);

            if (model.getIssueList().isSubtask(issueKey)) {
                var parentKey = issueData.parentKey;
                var isParentValid = model.getIssueList().isIssueValid(parentKey);
                var parentData = model.getIssueList().getIssueData(parentKey);
                var isParentCollapsed = SubtasksExpandingController.isIssueCollapsed(parentKey);

                // if parent is in the same column, collapsed and not filtered - issue should move even if its hidden
                if (isParentValid && isParentCollapsed && isIssueVisible(parentData)) {
                    return true;
                }
            }

            return isIssueVisible(issueData);
        });
    };

    /**
     * Remove from selection all issues that are in different model (sprint)
     * @param sprintId
     */
    BacklogSelectionController.removeFromSelectionIssuesInDifferentModel = function(sprintId) {
        var targetModel = BacklogModel.findTargetModel(sprintId);
        if (!targetModel) {
            return;
        }

        var selectedIssues = BacklogSelectionController.getSelectedIssueKeys();
        var issueList = targetModel.getIssueList();
        var issueKeysToUnselect = selectedIssues.filter(function (issueKey) {
            return !issueList.isIssueValid(issueKey);
        });

        if (issueKeysToUnselect.length > 0) {
            BacklogSelectionController.selectionManager.removeSelectedIssueKeys(issueKeysToUnselect);
            BacklogSelectionController.updateUIAndState({});
        }
    };

    return BacklogSelectionController;
});