AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common", "com.pyxis.greenhopper.jira:gh-rapid"], function () {
    var IssueListModel = require('jira-agile/rapid/ui/plan/issue-list-model');

    module("IssueListModelTests");

    test("module 'jira-agile/rapid/ui/plan/issue-list-model' exists", function () {
        ok(IssueListModel);
        ok(GH.IssueListModel);
    });

    test('Test visible issues', function () {
        var issues = GH.IssueTest.issueObjectsFromKeys(['GHS-1', 'GHS-2', 'GHS-3']);
        var fakeVersionModel = {
            getVersionsForIds: function getVersionsForIds() {}
        };
        var issueListModel = new IssueListModel('test', issues, fakeVersionModel);

        var visibleRankables = issueListModel.getVisibleRankables();
        equal(visibleRankables.length, 3, 'Expected 3 visible issues');
        equal(visibleRankables[0], 'GHS-1', 'Expected GHS-1 in visible issues');
        equal(visibleRankables[1], 'GHS-2', 'Expected GHS-2 in visible issues');
        equal(visibleRankables[2], 'GHS-3', 'Expected GHS-3 in visible issues');

        var issue = issueListModel.getIssueData('GHS-1');
        issue.hidden = true;
        issueListModel.updateIssue(issue);

        equal(issueListModel.getVisibleRankables().length, 2, 'Expected 2 visible issues');
        visibleRankables = issueListModel.getVisibleRankables();
        equal(visibleRankables[0], 'GHS-2', 'Expected GHS-2 in visible issues');
        equal(visibleRankables[1], 'GHS-3', 'Expected GHS-3 in visible issues');
    });

    test('Test issue validity', function () {
        var issues = GH.IssueTest.issueObjectsFromKeys(['GHS-1', 'GHS-2', 'GHS-3']);

        var issueListModel = new IssueListModel('test', issues);

        equal(issueListModel.isIssueValid('GHS-1'), true, 'Expected issue was valid');
        equal(issueListModel.isIssueValid('GHS-2'), true, 'Expected issue was valid');
        equal(issueListModel.isIssueValid('GHS-3'), true, 'Expected issue was valid');
        equal(issueListModel.isIssueValid('GHS-4'), false, 'Expected issue was invalid');

        issueListModel.removeIssues(['GHS-1']);

        equal(issueListModel.isIssueValid('GHS-1'), false, 'Expected issue was invalid');
    });

    test('Test inserting rankable keys', function () {
        var issues = GH.IssueTest.issueObjectsFromKeys(['GHS-1', 'GHS-2', 'GHS-3']);
        var issueListModel = new IssueListModel('test', issues);

        issueListModel.insertMarker(-5, 'GHS-1');

        equal(issueListModel.getRankableBeforeId('GHS-1'), false, 'expected GHS-1 was first');
        equal(issueListModel.getRankableBeforeId(-5), 'GHS-1', 'expected -5 was after GHS-1');
        equal(issueListModel.getRankableBeforeId('GHS-2'), -5, 'expected GHS-2 was after -5');

        issueListModel.insertMarker(-4, false);

        equal(issueListModel.getRankableBeforeId(-4), false, 'expected -4 was first');
        equal(issueListModel.getRankableBeforeId('GHS-1'), -4, 'expected GHS-1 was after -4');
        equal(issueListModel.getRankableBeforeId(-5), 'GHS-1', 'expected -5 was after GHS-1');
        equal(issueListModel.getRankableBeforeId('GHS-2'), -5, 'expected GHS-2 was after -5');

        issueListModel.insertMarker(-6, 'GHS-3');

        equal(issueListModel.getRankableBeforeId(-4), false, 'expected -4 was first');
        equal(issueListModel.getRankableBeforeId('GHS-1'), -4, 'expected GHS-1 was after -4');
        equal(issueListModel.getRankableBeforeId(-5), 'GHS-1', 'expected -5 was after GHS-1');
        equal(issueListModel.getRankableBeforeId('GHS-2'), -5, 'expected GHS-2 was after -5');
        equal(issueListModel.getRankableBeforeId('GHS-3'), 'GHS-2', 'expected GHS-3 was after GHS-2');
        equal(issueListModel.getRankableBeforeId(-6), 'GHS-3', 'expected -6 was after GHS-3');

        // reinsert -6 somewhere else
        issueListModel.insertMarker(-6, 'GHS-2');

        equal(issueListModel.getRankableBeforeId(-4), false, 'expected -4 was first');
        equal(issueListModel.getRankableBeforeId('GHS-1'), -4, 'expected GHS-1 was after -4');
        equal(issueListModel.getRankableBeforeId(-5), 'GHS-1', 'expected -5 was after GHS-1');
        equal(issueListModel.getRankableBeforeId('GHS-2'), -5, 'expected GHS-2 was after -5');
        equal(issueListModel.getRankableBeforeId(-6), 'GHS-2', 'expected -6 was after GHS-2');
        equal(issueListModel.getRankableBeforeId('GHS-3'), -6, 'expected GHS-3 was after -6');
    });

    test('Test issue validity', function () {
        var issues = GH.IssueTest.issueObjectsFromKeys(['GHS-1', 'GHS-2', 'GHS-3']);
        var issueListModel = new IssueListModel('test', issues);

        equal(issueListModel.areIssuesValid(['GHS-1']), true);
        equal(issueListModel.areIssuesValid(['GHS-1', 'GHS-2']), true);
        equal(issueListModel.areIssuesValid(['GHS-1', 'GHS-2', 'GHS-3']), true);
        equal(issueListModel.areIssuesValid(['GHS-1', 'GHS-2', 'GHS-3', 'GHS-4']), false);
        equal(issueListModel.areIssuesValid(['GHS-4']), false);
        equal(issueListModel.areIssuesValid([]), false);
    });

    test('Test canAddToSelection', function () {
        var issues = GH.IssueTest.issueObjectsFromKeys(['GHS-1', 'GHS-2', 'GHS-3']);
        issues[0].hidden = true;
        var issueListModel = new IssueListModel('test', issues);

        equal(issueListModel.canAddToSelection([], 'GHS-2'), true);
        equal(issueListModel.canAddToSelection(['GHS-1'], 'GHS-2'), false, 'Expected false because GHS-1 is hidden');
        equal(issueListModel.canAddToSelection(['GHS-2'], 'GHS-1'), false, 'Expected false because GHS-1 is hidden');
        equal(issueListModel.canAddToSelection(['GHS-2'], 'GHS-3'), true);
        equal(issueListModel.canAddToSelection(['GHS-4'], 'GHS-2'), false, 'Expected false because GHS-4 is not part of the list');
    });

    test('Test applySearchFilter', function () {
        var issues = GH.IssueTest.issueObjectsFromKeys(['GHS-1', 'GHS-2', 'GHS-3']);
        issues[0].hidden = true;
        var issueListModel = new IssueListModel('test', issues);

        var allIssuesShownSearchFilter = function allIssuesShownSearchFilter(issue) {
            return true;
        };
        var changed = issueListModel.applySearchFilter(allIssuesShownSearchFilter);
        equal(changed, false, 'Expected no changes');

        var allIssuesHiddenSearchFilter = function allIssuesHiddenSearchFilter(issue) {
            return false;
        };
        changed = issueListModel.applySearchFilter(allIssuesHiddenSearchFilter);
        equal(changed, true, 'Expected changes');
        equal(issueListModel.isIssueVisible('GHS-1'), false, 'Expected GHS-1 is hidden');
        equal(issueListModel.isIssueHiddenBySearchByKey('GHS-2'), true, 'Expected GHS-2 is hidden');
        equal(issueListModel.isIssueVisible('GHS-2'), false, 'Expected GHS-2 is hidden');
        equal(issueListModel.isIssueHiddenBySearchByKey('GHS-3'), true, 'Expected GHS-3 is hidden');
        equal(issueListModel.isIssueVisible('GHS-3'), false, 'Expected GHS-3 is hidden');

        var onlyShow3SearchFilter = function onlyShow3SearchFilter(issue) {
            return (/3/.test(issue.key)
            );
        };
        changed = issueListModel.applySearchFilter(onlyShow3SearchFilter);
        equal(changed, true, 'Expected changes');
        equal(issueListModel.isIssueVisible('GHS-1'), false, 'Expected GHS-1 is hidden');
        equal(issueListModel.isIssueHiddenBySearchByKey('GHS-2'), true, 'Expected GHS-2 is hidden');
        equal(issueListModel.isIssueVisible('GHS-2'), false, 'Expected GHS-2 is hidden');
        equal(issueListModel.isIssueHiddenBySearchByKey('GHS-3'), false, 'Expected GHS-3 is shown');
        equal(issueListModel.isIssueVisible('GHS-3'), true, 'Expected GHS-3 is shown');
    });

    test('Test getFirstRankableIssueKeyInColumn', function () {
        var issueList = new IssueListModel('test', GH.IssueTest.issueObjectsFromKeys(['GHS-3', 'GHS-2', 'GHS-1', 'GHS-4', 'GHS-5']));
        strictEqual(issueList.getFirstRankableIssueKeyInColumn(null), 'GHS-3', 'handles null');
        strictEqual(issueList.getFirstRankableIssueKeyInColumn('GHS-3'), 'GHS-2', 'handles string');
        strictEqual(issueList.getFirstRankableIssueKeyInColumn(['GHS-1', 'GHS-2']), 'GHS-3', 'handles array');
        strictEqual(issueList.getFirstRankableIssueKeyInColumn(['GHS-1', 'GHS-2', 'GHS-3']), 'GHS-4', 'handles array');
    });

    test('Test getFirstRankableIssueKeyInColumn with subtasks', function () {
        var issueList = new IssueListModel('test', GH.IssueTest.issueObjects([{ parentKey: 'GHS-PARENT', key: 'GHS-SUBTASK' }, 'GHS-3', 'GHS-2', 'GHS-1', 'GHS-4', 'GHS-5']));
        strictEqual(issueList.getFirstRankableIssueKeyInColumn([], true), 'GHS-PARENT', 'useParentIssue is true so parent is used');
        strictEqual(issueList.getFirstRankableIssueKeyInColumn([], false), 'GHS-SUBTASK', 'useParentIssue is false so subtask is used');
    });

    test('Test getLastRankableIssueKeyInColumn', function () {
        var issueList = new IssueListModel('test', GH.IssueTest.issueObjectsFromKeys(['GHS-3', 'GHS-2', 'GHS-1', 'GHS-4', 'GHS-5']));
        strictEqual(issueList.getLastRankableIssueKeyInColumn(null), 'GHS-5', 'handles null');
        strictEqual(issueList.getLastRankableIssueKeyInColumn('GHS-5'), 'GHS-4', 'handles string');
        strictEqual(issueList.getLastRankableIssueKeyInColumn(['GHS-1', 'GHS-4']), 'GHS-5', 'handles array');
        strictEqual(issueList.getLastRankableIssueKeyInColumn(['GHS-1', 'GHS-4', 'GHS-5']), 'GHS-2', 'handles array');
    });

    test('Test getLastRankableIssueKeyInColumn with subtasks', function () {
        var issueList = new IssueListModel('test', GH.IssueTest.issueObjects(['GHS-3', 'GHS-2', 'GHS-1', 'GHS-4', 'GHS-5', { parentKey: 'GHS-PARENT', key: 'GHS-SUBTASK' }]));
        strictEqual(issueList.getLastRankableIssueKeyInColumn([], true), 'GHS-PARENT', 'useParentIssue is true so parent is used');
        strictEqual(issueList.getLastRankableIssueKeyInColumn([], false), 'GHS-SUBTASK', 'useParentIssue is false so subtask is used');
    });

    test('Test areIssuesBetweenMarkers', function () {
        var issueList = new IssueListModel('test', GH.IssueTest.issueObjectsFromKeys(['GHS-3', 'GHS-2', 'GHS-1', 'GHS-4', 'GHS-5']));

        issueList.insertMarker(1, 'GHS-3');
        issueList.insertMarker(2, 'GHS-4');

        ok(issueList.areIssuesBetweenMarkers(['GHS-2', 'GHS-1', 'GHS-4'], 1, 2));
        ok(issueList.areIssuesBetweenMarkers(['GHS-2', 'GHS-1', 'GHS-4'], 2, 1));
        ok(issueList.areIssuesBetweenMarkers(['GHS-2', 'GHS-4'], 1, 2));
        ok(issueList.areIssuesBetweenMarkers(['GHS-2', 'GHS-1'], 1, 2));
        ok(issueList.areIssuesBetweenMarkers(['GHS-1'], 1, 2));
        ok(!issueList.areIssuesBetweenMarkers(['GHS-3', 'GHS-2', 'GHS-1', 'GHS-4'], 1, 2));
        ok(!issueList.areIssuesBetweenMarkers(['GHS-3', 'GHS-2', 'GHS-1', 'GHS-4', 'GHS-5'], 1, 2));
        ok(!issueList.areIssuesBetweenMarkers(['GHS-3', 'GHS-5'], 1, 2));
        ok(!issueList.areIssuesBetweenMarkers(['GHS-4', 'GHS-5'], 1, 2));
        ok(!issueList.areIssuesBetweenMarkers(['GHS-4', 'GHS-5'], 88, 99));
    });

    test('Test getMarkerBeforeId', function () {
        var issueList = new IssueListModel('test', GH.IssueTest.issueObjectsFromKeys(['GHS-3', 'GHS-2', 'GHS-1', 'GHS-4', 'GHS-5']));
        issueList.insertMarker(0, false);
        issueList.insertMarker(1, 'GHS-3');
        issueList.insertMarker(2, 'GHS-4');
        issueList.insertMarker(3, 'GHS-5');

        strictEqual(issueList.getMarkerIdBeforeId(1), 0, 'Expected marker 0 to be returned');
        strictEqual(issueList.getMarkerIdBeforeId(2), 1, 'Expected marker 1 to be returned');
        strictEqual(issueList.getMarkerIdBeforeId(3), 2, 'Expected marker 2 to be returned');
        ok(!issueList.getMarkerIdBeforeId(0));
    });

    test('Test getIssueCount', function () {
        var issueList = new IssueListModel('test', GH.IssueTest.issueObjectsFromKeys(['GHS-3', 'GHS-2', 'GHS-1', 'GHS-4', 'GHS-5']));
        issueList.insertMarker(0, false);
        issueList.insertMarker(1, 'GHS-3');
        issueList.insertMarker(2, 'GHS-4');
        issueList.insertMarker(3, 'GHS-5');

        strictEqual(issueList.getIssueCount(), 5, 'Expected an issue count of 5');
    });

    test('Test getLastMarkerId', function () {
        var issueList = new IssueListModel('test', GH.IssueTest.issueObjectsFromKeys(['GHS-3', 'GHS-2', 'GHS-1', 'GHS-4', 'GHS-5']));
        issueList.insertMarker(0, false);
        issueList.insertMarker(1, 'GHS-3');
        issueList.insertMarker(2, 'GHS-4');
        issueList.insertMarker(3, 'GHS-5');

        strictEqual(issueList.getLastMarkerId(), 3, 'Expected the last marker id to be 3');
    });

    test('Test getMarkerIds', function () {
        var issueList = new IssueListModel('test', GH.IssueTest.issueObjectsFromKeys(['GHS-3', 'GHS-2', 'GHS-1', 'GHS-4', 'GHS-5']));
        issueList.insertMarker(0, false);
        issueList.insertMarker(1, 'GHS-3');
        issueList.insertMarker(2, 'GHS-4');

        strictEqual(issueList.getMarkerIds().length, 3, 'Expected 3 markers to be returned');

        issueList.insertMarker(3, 'GHS-5');

        strictEqual(issueList.getMarkerIds().length, 4, 'Expected 4 markers to be returned');
        deepEqual(issueList.getMarkerIds(), [0, 1, 2, 3], 'Expected an array with marker ids [0,1,2,3]');
    });

    function createIssue(key) {
        return {
            key: key
        };
    }

    function createSubTask(parentKey, key) {
        return {
            key: key,
            parentKey: parentKey
        };
    }

    module('IssueListModel reorderIssues', {
        setup: function setup() {
            /**
             * Subtasks are S-# and parents are P-# for readability.
             * Obviously in a real instance the project key would be the same for parents and their sub-tasks.
             */
            var issues = [createIssue('P-1'), createSubTask('P-1', 'S-2'), createSubTask('P-1', 'S-3'), createIssue('P-4'), createIssue('P-5'), createSubTask('P-5', 'S-6'), createIssue('P-7'),

            // orphan of fake parent P-8
            createSubTask('P-8', 'S-9')];

            var issueListModel = new IssueListModel('test-model', issues);

            var afterKey = undefined;
            var beforeKey = undefined;
            var issueKeysToRank = [];

            this.given = {};

            this.given.userRanksIssues = function (issueKeys) {
                issueKeysToRank = issueKeys;
                return {
                    beforeIssue: function beforeIssue(key) {
                        return beforeKey = key;
                    },
                    afterIssue: function afterIssue(key) {
                        return afterKey = key;
                    }
                };
            };

            this.when = {};

            this.when.rankingPerformed = function () {
                return issueListModel.reorderIssues(issueKeysToRank, afterKey, beforeKey);
            };

            this.verify = {};

            this.verify.issueOrderIs = function (expectedOrder, message) {
                var resultOrder = issueListModel.getIssuesInOrder().map(function (issue) {
                    return issue.key;
                });
                return deepEqual(resultOrder, expectedOrder, message);
            };
        }
    });

    test('can move issue after another issue', function () {
        this.given.userRanksIssues(['P-4']).afterIssue('P-7');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-1', 'S-2', 'S-3', 'P-5', 'S-6', 'P-7', // <- rankAfter
        'P-4', // <- moved issue
        'S-9'], 'Expected P-4 to be moved to after P-7');
    });

    test('can move issue before another issue', function () {
        this.given.userRanksIssues(['P-7']).beforeIssue('P-4');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-1', 'S-2', 'S-3', 'P-7', // <- moved issue
        'P-4', // <- rankBefore
        'P-5', 'S-6', 'S-9'], 'Expected P-7 to be moved to before P-4');
    });

    test('moving parent with rankBefore moves children', function () {
        this.given.userRanksIssues(['P-5']).beforeIssue('P-4');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-1', 'S-2', 'S-3', 'P-5', // <- moved issue with children
        'S-6', 'P-4', // <- rankBefore
        'P-7', 'S-9'], 'Expected P-5 and children to be ranked before P-4');
    });

    test('moving parent with rankAfter moves children', function () {
        this.given.userRanksIssues(['P-1']).afterIssue('P-4');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-4', // <- rank after
        'P-1', // <- moved issue with children
        'S-2', 'S-3', 'P-5', 'S-6', 'P-7', 'S-9'], 'Expected P-1 and children to be ranked after P-4');
    });

    test('moving parent with rankAfter inserts after children of rankAfter parent', function () {
        this.given.userRanksIssues(['P-1']).afterIssue('P-5');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-4', 'P-5', // <- rank after (ranks after last sub-task)
        'S-6', 'P-1', // <- moved issue with children
        'S-2', 'S-3', 'P-7', 'S-9'], 'Expected P-1 and children to be ranked after P-5 and children');
    });

    test('can move a parent after an orphaned sub-task', function () {
        this.given.userRanksIssues(['P-7']).afterIssue('P-8');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-1', 'S-2', 'S-3', 'P-4', 'P-5', 'S-6', 'S-9', // <- rankAfter P-8
        'P-7' // <- moved issue
        ], 'Expected P-7 to be ranked after S-9, the orphan child of P-8');
    });

    test('can move a parent before an orphaned sub-task', function () {
        this.given.userRanksIssues(['P-1']).beforeIssue('P-8');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-4', 'P-5', 'S-6', 'P-7', 'P-1', // <- moved issue with children
        'S-2', 'S-3', 'S-9' // <- rankBefore P-8
        ], 'Expected P-1 and children to be ranked before S-9, the orphan child of P-8');
    });

    test('can move sub-task after another in the same parent', function () {
        this.given.userRanksIssues(['S-2']).afterIssue('S-3');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-1', 'S-3', // <- rankAfter
        'S-2', // <- moved sub-task
        'P-4', 'P-5', 'S-6', 'P-7', 'S-9'], 'Expected S-3 to be ranked above S-2');
    });

    test('can move sub-task before another in the same parent', function () {
        this.given.userRanksIssues(['S-3']).beforeIssue('S-2');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-1', 'S-3', // <- rankAfter
        'S-2', // <- moved sub-task
        'P-4', 'P-5', 'S-6', 'P-7', 'S-9'], 'Expected S-2 to be ranked below S-3');
    });

    test('ignores selected subTasks and just moves all parents with children when ranking against a parent', function () {
        this.given.userRanksIssues(['P-1', 'S-2', 'S-3', 'P-5', 'S-6', 'S-9']).beforeIssue('P-4');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-1', // <- selection start
        'S-2', 'S-3', 'P-5', 'S-6', // <- selection end
        'P-4', // <- rankBefore
        'P-7', 'S-9' // <- did not move
        ], 'Expected S-9 to not move, and P-1, P-5 (with children) to be ranked before P-4');
    });

    test('ignores unselected subTasks in a multi-select and still moves all children, when ranking against a parent', function () {
        this.given.userRanksIssues(['P-1', 'S-3']).afterIssue('P-4');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-4', // <- rankAfter
        'P-1', // <- selected
        'S-2', // <- was not selected, but still moved
        'S-3', // <- selected
        'P-5', 'S-6', 'P-7', 'S-9'], 'Expected P-1 and all children to be ranked after P-4');
    });

    test('ignores selected parents in a multi-select, and just moves subTasks, when ranking against another subTask', function () {
        this.given.userRanksIssues(['S-3', 'P-4']).beforeIssue('S-2');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-1', 'S-3', // <- selected
        'S-2', // <- rankBefore
        'P-4', // <- selected, but not moved
        'P-5', 'S-6', 'P-7', 'S-9'], 'Expected S-3 to be ranked before S-2, and P-4 to not be moved.');
    });

    test('ignores subTasks from a different parent in a multi-select, and only moves subTasks that are siblings of the ranked against subTask', function () {
        this.given.userRanksIssues(['S-3', 'S-6']).beforeIssue('S-2');

        this.when.rankingPerformed();

        this.verify.issueOrderIs(['P-1', 'S-3', // <- selected
        'S-2', // <- rankBefore
        'P-4', 'P-5', 'S-6', // <- selected, but not moved
        'P-7', 'S-9'], 'Expected S-3 to be ranked before S-2, and P-6 to not be moved.');
    });

    module('getIssuesExcludingSubtasks');

    test('getIssuesExcludingSubtasks() should exclude subTasks', function () {
        var issueListModel = new IssueListModel('model', [GH.IssueTest.createIssueObject({ id: 1, 'hidden': false }), GH.IssueTest.createIssueObject({ id: 2, 'hidden': false, parentId: 1 }), GH.IssueTest.createIssueObject({ id: 3, 'hidden': true })]);
        var issues = issueListModel.getIssuesExcludingSubtasks();
        equal(issues.length, 2);
    });
});