/* globals
 * GH, GH.VersionController, GH.EpicController, GH.Logger, GH.DetailsView, GH.RapidBoard,
 * GH.planOnboarding, GH.RapidViewConfig, GH.ViewActions, GH.BoardState
 */

/**
 * Plan Controller
 * @module jira-agile/rapid/ui/plan/plan-controller
 * @requires module:jquery
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-selection-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-controls
 * @requires module:jira-agile/rapid/ui/plan/plan-issue-list-filtering
 * @requires module:jira-agile/rapid/ui/plan/plan-view
 */
define('jira-agile/rapid/ui/plan/plan-controller', ['require'], function (require) {
    'use strict';

    var $ = require('jquery');
    var _ = require('underscore');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var AnalyticsHelper = require('jira-agile/rapid/ui/plan/analytics-helper');
    var PlanControls = require('jira-agile/rapid/ui/plan/plan-controls');
    var PlanIssueListFiltering = require('jira-agile/rapid/ui/plan/plan-issue-list-filtering');
    var PlanView = require('jira-agile/rapid/ui/plan/plan-view');
    var KanPlanFeatureService = require('jira-agile/rapid/ui/kanplan/kan-plan-feature-service');
    var KanPlanInlineDialogController = require('jira-agile/rapid/ui/kanplan/KanPlanInlineDialogController');
    var BacklogSelectionController;
    var BacklogController;

    // Resolve circular dependencies
    GlobalEvents.on('pre-initialization', function () {
        BacklogSelectionController = require('jira-agile/rapid/ui/plan/backlog-selection-controller');
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
    });

    var PlanController = {};

    PlanController.isVisible = false;

    PlanController.viewData = undefined;
    PlanController.currentViewConfig = {};

    PlanController.analytics = function (categorySuffix) {
        return new AnalyticsTracker('gh.planningboard.' + categorySuffix);
    };

    /**
     * Initializes the view controller.
     */
    PlanController.init = _.once(function () {
        // initialize the view
        PlanView.init();

        // init controls (quick filters, etc)
        PlanControls.init();

        // initialize the version controller
        GH.VersionController.init();

        // initialise epics
        GH.EpicController.init();

        // initialise the backlog
        BacklogController.init();

        KanPlanInlineDialogController.init();
    });

    /**
     * Is the plan controller currently active
     * @return {boolean}
     */
    PlanController.isActive = function () {
        return PlanController.isVisible;
    };

    /**
     * Sets the rapid view data.
     */
    PlanController.setRapidView = function (rapidViewData) {
        PlanController.rapidViewData = rapidViewData;

        // set data on sub-widgets
        BacklogController.setRapidView(rapidViewData);
    };

    /**
     * Called when the plan tab is shown
     */
    PlanController.show = function () {
        GH.log("PlanController.show", GH.Logger.Contexts.ui);
        if (!PlanController.rapidViewData) {
            return;
        }
        PlanController.isVisible = true;

        // show the view
        PlanView.show();

        // configure the UI
        PlanController.configureUI();

        // kick off loading the view configuration
        PlanController.loadRapidViewConfig();

        PlanController.attachControlFHandler();
    };

    PlanController.configureUI = function () {
        // fetch the selected issue
        var selectedIssueKey = BacklogSelectionController.getSelectedIssueKey();

        // set the correct rapid view id
        GH.DetailsView.setRapidViewId(PlanController.rapidViewData.id);

        if (selectedIssueKey) {
            // tell the detail view what issue to load.
            GH.DetailsView.setSelectedIssueKey(selectedIssueKey);
        }

        PlanView.updateDetailViewState();

        // toggle the version column
        PlanView.updateVersionColumnState();

        // toggle the epics column
        PlanView.updateEpicsColumnState();
    };

    PlanController.lastPushUpdate = 0;
    PlanController.pushTimeoutHandler = undefined;
    PlanController.updateDetailsView = function (issueKey, forceShow, detailsViewReloadReason) {
        var stallDuration = 300;
        var now = new Date().getTime();

        // update directly if older than x ms
        if (PlanController.lastPushUpdate < (now - stallDuration)) {
            PlanController.doUpdateDetailsView(issueKey, forceShow, detailsViewReloadReason);
            PlanController.lastPushUpdate = now + stallDuration;
            return;
        }

        // clear previous timeout handler
        if (PlanController.pushTimeoutHandler) {
            window.clearTimeout(PlanController.pushTimeoutHandler);
        }
        // register new handler
        PlanController.pushTimeoutHandler = window.setTimeout(function () {
            PlanController.doUpdateDetailsView(issueKey, forceShow, detailsViewReloadReason);
            PlanController.lastPushUpdate = new Date().getTime();
        }, stallDuration);
    };

    PlanController.doUpdateDetailsView = function (issueKey, forceShow, detailsViewReloadReason) {
        if (!GH.RapidBoard.State.isDetailsViewPlanOpened() && !forceShow) {
            return;
        }
        // set the selected issue key
        GH.DetailsView.setSelectedIssueKey(issueKey);

        if (BacklogSelectionController.getSelectedIssueKey() == null) {
            PlanController.setDetailViewOpenedState(false);
            GH.DetailsView.hide();
        } else {
            PlanController.setDetailViewOpenedState(true);
            GH.DetailsView.show(detailsViewReloadReason);
        }

        // assume state has changed - push state. This is not the same as how Work Mode does it -
        // they check to see if the selected issue changed
        if (issueKey) {
            setTimeout(function () {
                GH.RapidBoard.State.pushState();
            }, 0);
        }

        // ensure the detail view is properly laid out (this is only necessary the first time after we show the tab,
        // after that resize handlers will take care of re-layouting)
        PlanView.doUpdateDetailsView();
    };

    PlanController.reloadDetailView = function (detailsViewLoadReason) {
        // ignore if not visible
        if (!PlanController.isDetailsViewOpened()) {
            return;
        }

        PlanController.setDetailViewOpenedState(true);
        GH.DetailsView.show(detailsViewLoadReason);
    };

    PlanController.reload = function (callback) {
        if (!PlanController.isVisible) {
            return;
        }
        BacklogController.show(callback);
    };

    /**
     * Called when the plan tab is hidden
     */
    PlanController.hide = function () {
        GH.log("PlanController.hide", GH.Logger.Contexts.ui);
        PlanController.isVisible = false;

        // hide sub-widgets
        BacklogController.hide();
        GH.DetailsView.hide();

        PlanController.detachControlFHandler();

        // hide the plan view
        PlanView.hide();

        // hide onboarding chaperones
        GH.planOnboarding.refresh();
    };

    /**
     * Load the rapid view configuration used by the pool
     */
    PlanController.loadRapidViewConfig = function () {
        // callback called when the view configuration has been loaded
        var callback = function (data) {
            PlanController.currentViewConfig = data;

            if (GH.RapidBoard.State.isScrumBoard() || KanPlanFeatureService.shouldShowKanbanBacklog()) {
                PlanController.initializePlanMode();
            }
        };

        // fetch the configuration
        GH.RapidViewConfig.fetchConfiguration(PlanController.rapidViewData.id).done(callback);
    };

    /**
     * Initialize the pool
     */
    PlanController.initializePlanMode = function () {
        // don't continue if we are not visible
        if (!PlanController.isVisible) {
            return;
        }

        PlanControls.setRapidViewConfig(PlanController.currentViewConfig);
        BacklogController.setRapidViewConfig(PlanController.currentViewConfig);
        GH.DetailsView.setRapidViewConfig(PlanController.currentViewConfig);

        // render the plan controls
        PlanControls.renderControls();

        // render subnavigator, which in this case only contains the header
        if (GH.RapidBoard.ViewController.shouldShowSubnavigator()) {
            GH.RapidBoard.SubnavigatorController.render();
        }

        // hide the loading filters transition
        PlanView.hideLoadingFilter();

        // clear last executed instant filter (when switching between modes)
        PlanIssueListFiltering.setInstantFilter(null);

        // show Backlog
        BacklogController.show(emitPlanModeViewAnalytic);

        GH.RapidBoard.ViewController.updateContentContainer();

    };

    var emitPlanModeViewAnalytic = function () {
        var eventName = 'jira-software.' + GH.RapidBoard.State.getBoardType() + '.plan.view';
        var data = AnalyticsHelper.baseEventData();
        KanPlanFeatureService.sendAnalyticsInfo(eventName, data);
    };

    /**
     * Is the detail view opened
     */
    PlanController.isDetailsViewOpened = function () {
        return GH.RapidBoard.State.isDetailsViewPlanOpened();
    };

    /**
     * Set the new detail view opened state
     */
    PlanController.setDetailViewOpenedState = function (opened) {
        opened = opened || false;
        GH.RapidBoard.State.setPlanViewDetailsViewOpened(opened);

        PlanView.updateDetailViewState();
    };

    /**
     * Closes the detail view.
     */
    PlanController.closeDetailsView = function closeDetailsView () {
        // ignore if there are any errors
        if (GH.DetailsView.hasCurrentError()) {
            return;
        }

        // ignore if already closed
        if (!PlanController.isDetailsViewOpened()) {
            return;
        }

        // mark the details view as closed and redraw
        PlanController.setDetailViewOpenedState(false);
        GH.RapidBoard.State.pushState();

        // update the view
        GH.DetailsView.hide();

        PlanView.updateHorizontalPositioning();
    };

    /**
     * Toggles the details view
     */
    PlanController.toggleDetailsView = function () {
        if (PlanController.isDetailsViewOpened()) {
            PlanController.closeDetailsView();
        } else {
            // fetch the issue key to display, none is valid on work mode as well
            var selectedIssueKey = BacklogSelectionController.getSelectedIssueKey();
            PlanController.updateDetailsView(selectedIssueKey, true);
        }
    };

    /**
     * Toggles the epic lozenges showing on the cards
     */
    PlanController.toggleEpicsShowOnBacklog = function () {
        GH.ViewActions.toggleEpicsShowOnRapidBoard();
        PlanController.reload();
    };

    /**
     * Is the epics panel enabled or not (true for scrum board, depends on feature flags and epic switch state for kanban board)
     */
    PlanController.isEpicsPanelEnabled = function () {
        return GH.RapidBoard.State.isScrumBoard() || KanPlanFeatureService.shouldShowEpicsPanel()
    };

    /**
     * Is the versions panel enabled or not (true for scrum board, depends on feature flags for kanban board)
     */
    PlanController.isVersionsPanelEnabled = function () {
        return GH.RapidBoard.State.isScrumBoard() || KanPlanFeatureService.isEpicsAndVersionsEnabled()
    };

    // Versions column open state
    PlanController.isVersionsColumnVisible = function () {
        return GH.BoardState.getPerViewValue('planVersionsVisible', false);
    };

    // Epics column open state
    PlanController.isEpicsColumnVisible = function () {
        return GH.BoardState.getPerViewValue('planEpicsVisible', false);
    };

    PlanController.isVersionsColumnVisiblePersisted = function () {
        // by default, we want the version column hidden
        return GH.BoardState.getPerViewValue('planVersionsVisible', false);
    };

    // used by the PlanUrlState, as it is invoked before epicsSupport flag is properly loaded
    PlanController.isEpicsColumnVisiblePersisted = function () {
        return GH.BoardState.getPerViewValue('planEpicsVisible', false);
    };

    PlanController.setVersionsColumnVisible = function (visible) {
        GH.BoardState.setPerViewValue('planVersionsVisible', !!visible);
    };

    PlanController.setEpicsColumnVisible = function (visible) {
        GH.BoardState.setPerViewValue('planEpicsVisible', !!visible);
    };

    PlanController.toggleVersionsColumn = function () {
        var visible = PlanController.isVersionsColumnVisible();
        visible = !visible;
        PlanController.setVersionsColumnVisible(visible);
        GH.RapidBoard.State.pushState();

        GH.VersionController.sendAnalytics('toggleVersionsColumn', {"versionsColumnVisible": visible});

        // update the column state
        PlanView.updateVersionColumnState();

        // updated horizontal positioning to get detail view update
        PlanView.updateHorizontalPositioning();
        KanPlanInlineDialogController.trigger('updateepicsonboardingposition', PlanController.isEpicsColumnVisible());
    };

    PlanController.toggleEpicsColumn = function () {
        var visible = PlanController.isEpicsColumnVisible();
        visible = !visible;
        PlanController.setEpicsColumnVisible(visible);
        GH.RapidBoard.State.pushState();

        GH.EpicController.sendAnalytics('toggleEpicsColumn', {"epicsColumnVisible": visible});

        // update the column state
        PlanView.updateEpicsColumnState();

        // updated horizontal positioning to get detail view update
        PlanView.updateHorizontalPositioning();
        KanPlanInlineDialogController.trigger('updateepicsonboardingposition', visible);
    };

    PlanController.attachControlFHandler = function () {
        $(document).bind('keydown', PlanController.controlFHandler);
    };

    PlanController.detachControlFHandler = function () {
        $(document).unbind('keydown', PlanController.controlFHandler);
    };

    PlanController.controlFHandler = function (e) {
        if (e.ctrlKey && String.fromCharCode(e.which).toLowerCase() === "f") {
            PlanController.analytics('shortcuts').trigger('controlf'); // SAFE
        }
    };

    PlanController.collectPrintableIssues = function () {
        return {
            sprintIssues: BacklogController.extractPrintableIssuesFromAllSprints(),
            backlogIssues: BacklogController.getPrintableIssuesFromPlanBacklog(),
            epics: BacklogController.getEpicMap(),
            viewMode: 'plan',
            boardType: GH.RapidBoard.State.getBoardType()
        };
    };

    return PlanController;
});
