/* globals
 * GH.Notification
 */

/**
 * @module jira-agile/rapid/ui/plan/plan-kb-shortcuts
 * @requires module:underscore
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira-agile/rapid/ui/plan/backlog-model
 * @requires module:jira-agile/rapid/ui/plan/backlog-selection-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-drag-and-drop
 * @requires module:jira-agile/rapid/ui/plan/sprint-model
 */
define('jira-agile/rapid/ui/plan/plan-kb-shortcuts', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogSelectionController = require('jira-agile/rapid/ui/plan/backlog-selection-controller');
    var PlanController = require('jira-agile/rapid/ui/plan/plan-controller');
    var PlanDragAndDrop = require('jira-agile/rapid/ui/plan/plan-drag-and-drop');
    var SprintModel = require('jira-agile/rapid/ui/plan/sprint-model');
    var SubtasksExpandingController = require('jira-agile/rapid/ui/plan/subtasks-expanding-controller');

    var PlanKBShortcuts = {};

    PlanKBShortcuts.analytics = new AnalyticsTracker('gh.rapidboard.rankissues');

    function hasVisibleSubtask(model, selectedIssuesInOrder) {
        return _.any(selectedIssuesInOrder, function (key) {
            var issueList = model.getIssueList();
            var issue = issueList.getIssueData(key);
            if (issueList.isSubtask(key)) {
                // If a subtask is in selection, it's parent should be in the same column and collapsed
                if (issueList.isIssueValid(issue.parentKey)) {
                    var parentCollapsed = SubtasksExpandingController.isIssueCollapsed(issue.parentKey);
                    return !parentCollapsed;
                }
                return true;
            }
        });
    }

    /**
     * Re-ranks the selected issue(s) so that they are at the top of their swimlane-column.
     */
    PlanKBShortcuts.sendToTop = function (source) {
        // we don't want to do anything if the current view is not rankable
        if (!BacklogModel.isRankable()) {
            return;
        }

        // get the currently selected issue
        var issueKey = BacklogSelectionController.getSelectedIssueKey();
        if (!issueKey) {
            return;
        }

        // get the issue which we will be ranking before (either hidden by search or visible)
        var firstIssueKey = BacklogModel.getFirstRankableIssueKeyInColumn(issueKey, true);

        // get the multi-selected issues, in currently sorted order.
        var selectedIssuesInOrder = BacklogSelectionController.getSelectedAndVisibleIssuesInOrder();

        // do analytics before we even know if they are successful
        PlanKBShortcuts.analytics.trigger('sendToTop' + (source ? source : ''), selectedIssuesInOrder.length); // SAFE

        if (!firstIssueKey) {
            return;
        }

        // Resolve context of the selected issues (in backlog or in sprint list)
        var model = BacklogModel.findModelWithIssue(issueKey);

        if (hasVisibleSubtask(model, selectedIssuesInOrder)) {
            return;
        }

        // get a handle of the issue to select after we re-rank
        // this will be the issue that is going INTO our position - i.e. the previous issue
        var issueKeyToSelectAfterRanking = BacklogModel.getBestFitSelectionAfterRank(selectedIssuesInOrder);

        // select the issue if we got one
        if (issueKeyToSelectAfterRanking) {
            BacklogSelectionController.selectIssue(issueKeyToSelectAfterRanking);
        }

        var isSprintModel = SprintModel.isSprintModel(model);
        var sprintId = isSprintModel ? model.getSprintId() : null;

        // do the re-ranking
        PlanDragAndDrop.rankIssues(selectedIssuesInOrder, sprintId, null, firstIssueKey).done(function (data) {
            // show a success message
            GH.Notification.showSuccess(AJS.I18n.getText("gh.keyboard.shortcut.send.to.top.success"));
        }).fail(function () {
            // if ranking failed, reload model and redraw
            PlanController.reload();
        });
    };

    /**
     * Re-ranks the selected issue(s) so that they are at the bottom of their swimlane-column.
     */
    PlanKBShortcuts.sendToBottom = function (source) {
        // we don't want to do anything if the current view is not rankable
        if (!BacklogModel.isRankable()) {
            return;
        }

        // get the currently selected issue
        var issueKey = BacklogSelectionController.getSelectedIssueKey();
        if (!issueKey) {
            return;
        }

        // get the issue which we will be ranking before
        var lastIssueKey = BacklogModel.getLastRankableIssueKeyInColumn(issueKey, true);

        // get the multi-selected issues, in currently sorted order.
        var selectedIssuesInOrder = BacklogSelectionController.getSelectedAndVisibleIssuesInOrder();

        // do analytics before we even know if they are successful
        PlanKBShortcuts.analytics.trigger('sendToBottom' + (source ? source : ''), selectedIssuesInOrder.length); // SAFE

        if (!lastIssueKey) {
            return;
        }

        // Resolve context of the selected issues (in backlog or in sprint list)
        var model = BacklogModel.findModelWithIssue(issueKey);

        if (hasVisibleSubtask(model, selectedIssuesInOrder)) {
            return;
        }

        // get a handle of the issue to select after we re-rank
        // this will be the issue that is going INTO our position - i.e. the previous issue
        var issueKeyToSelectAfterRanking = BacklogModel.getBestFitSelectionAfterRank(selectedIssuesInOrder);

        // select the issue if we got one
        if (issueKeyToSelectAfterRanking) {
            BacklogSelectionController.selectIssue(issueKeyToSelectAfterRanking);
        }

        var isSprintModel = SprintModel.isSprintModel(model);
        var sprintId = isSprintModel ? model.getSprintId() : null;

        // do the re-ranking
        PlanDragAndDrop.rankIssues(selectedIssuesInOrder, sprintId, lastIssueKey, null).done(function (data) {
            // show a success message
            GH.Notification.showSuccess(AJS.I18n.getText("gh.keyboard.shortcut.send.to.bottom.success"));
        }).fail(function () {
            // if ranking failed, reload model and redraw
            PlanController.reload();
        });
    };

    return PlanKBShortcuts;
});