/* globals
 * GH.RankController
 */

/**
 * @module jira-agile/rapid/ui/plan/plan-rank-controller
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-model
 * @requires module:jira-agile/rapid/ui/plan/backlog-model2
 */
define('jira-agile/rapid/ui/plan/plan-rank-controller', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var $ = require('jquery');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var BacklogController;
    var BacklogSelectionController = require('jira-agile/rapid/ui/plan/backlog-selection-controller');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogModel2 = require('jira-agile/rapid/ui/plan/backlog-model2');

    // Resolve circular dependency
    GlobalEvents.on('pre-initialization', function() {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
    });

    var PlanRankController = {};

    /**
     * Initializes the controller
     */
    PlanRankController.init = function() {
        BacklogController.registerIssueCreatedCallback("rankIssues", PlanRankController.rankCreatedIssues);
    };

    /**
     * Handler for issue creation events. Ranks the issue at the most appropriate location,
     * either after the current selection or
     */
    PlanRankController.rankCreatedIssues = function(issues) {
        var issueKeys = BacklogSelectionController.getSelectedIssuesInOrder();
        if(issueKeys && issueKeys.length > 0) {
            return PlanRankController.rankCreatedIssuesBySelection(issues, issueKeys);
        } else {
            return PlanRankController.rankCreatedIssuesByVisibility(issues);
        }
    };

    /**
     * Ranks issues after the selected issues
     */
    PlanRankController.rankCreatedIssuesBySelection = function(issues, selectedIssueKeys) {
        var targetIssueKey = _.last(selectedIssueKeys);

        // get the model for the first issue, we only rank inside the backlog!
        var model = BacklogModel.findModelWithIssue(_.first(selectedIssueKeys));
        var issueKey;
        var inverted;
        if (model && BacklogModel2.isBacklogModel(model)) {
            issueKey = targetIssueKey;
            inverted = false;
        } else {
            // we can only rank into the backlog. find first issue
            issueKey = BacklogModel.getBacklogModel2().getIssueList().getFirstRankableIssueKeyInColumn(null, true);
            inverted = true;
        }
        if (issueKey) {
            return PlanRankController.rankCreatedIssuesAgainstIssue(issues, issueKey, inverted);
        }
    };

    /**
     * Ranks issues according to the current issue visibility
     */
    PlanRankController.rankCreatedIssuesByVisibility = function(issues) {
        var backlogContainer = $(".ghx-backlog-group");
        var backlogIssues = $(".js-issue", backlogContainer);

        var windowHeight = $(document).height();
        var scrollPosition = $(document).scrollTop();
        function isVisible(el) {
            var elOffset = el.offset().top;
            return elOffset > scrollPosition && elOffset < windowHeight + scrollPosition;
        }

        var visibleBacklogIssues = backlogIssues.filter(function() {
            var el = $(this);
            return isVisible(el);
        });

        var issueElementToRankAgainst;
        if(visibleBacklogIssues.length > 0) {
            issueElementToRankAgainst = visibleBacklogIssues.first();
        } else {
            issueElementToRankAgainst = backlogIssues.first();
        }

        if(issueElementToRankAgainst && issueElementToRankAgainst.length === 1) {
            var issueKeyToRankAgainst = issueElementToRankAgainst.attr("data-issue-key");
            return PlanRankController.rankCreatedIssuesAgainstIssue(issues, issueKeyToRankAgainst, true);
        }
    };

    PlanRankController.rankCreatedIssuesAgainstIssue = function(issues, targetIssueKey, before) {
        var issueKeys = _.pluck(issues, 'issueKey');
        var invertedKeys = issueKeys.reverse();

        return GH.RankController.rankIssues(BacklogModel.getRankCustomFieldId(), invertedKeys,
            before ? targetIssueKey : null, before ? null : targetIssueKey, {deferErrorHandling:true});
    };

    return PlanRankController;
});