/* globals
 * GH.Logger, GH.LinkedPagesController, GH.LinkedPagesController, GH.Tooltip,
 * GH.EpicController, GH.RapidBoard, GH.tpl
 */

/**
 * This module representing SprintView
 * @module jira-agile/rapid/ui/plan/sprint-view
 * @requires module:underscore
 * @requires module:jquery
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira-agile/rapid/ui/plan/sprint-model
 * @requires module:jira-agile/rapid/ui/plan/sprint-controller
 * @requires module:jira-agile/rapid/ui/plan/sprint-controller
 * @requires module:jira-agile/rapid/ui/work/issue-list-util
 * @requires module:jira-agile/rapid/ui/plan/backlog-statistics
 * @requires module:jira-agile/rapid/ui/plan/issue-list-util
 * @requires module:jira-agile/rapid/ui/plan/subtasks-expanding-controller
 * @requires module:jira/featureflags/feature-manager
 * @requires module:jira-help-tips/feature/help-tip
 * @requires module:jira/util/formatter
 * @requires module:wrm/data
 */
define('jira-agile/rapid/ui/plan/sprint-view', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var $ = require('jquery');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var SprintModel = require('jira-agile/rapid/ui/plan/sprint-model');
    var AssignedWorkDialog = require('jira-agile/rapid/ui/plan/assigned-work-dialog');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogStatistics = require('jira-agile/rapid/ui/plan/backlog-statistics');
    var IssueListUtil = require('jira-agile/rapid/ui/plan/issue-list-util');
    var SubtasksExpandingController = require('jira-agile/rapid/ui/plan/subtasks-expanding-controller');
    var FeatureFlagManager = require('jira/featureflags/feature-manager');
    var HelpTip = require('jira-help-tips/feature/help-tip');
    var formatter = require('jira/util/formatter');
    var wrmData = require('wrm/data');
    var SprintController;
    var BacklogController;

    var DARK_FEATURE_SPRINT_GOAL_KEY = "com.atlassian.jira.agile.darkfeature.sprint.goal";

    // Resolve circular dependency
    GlobalEvents.on('pre-initialization', function () {
        SprintController = require('jira-agile/rapid/ui/plan/sprint-controller');
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
    });

    /**
     /**
     * View for sprints
     */
    var SprintView = {};

    SprintView.sprintGoalOnboardingState = wrmData.claim("com.pyxis.greenhopper.jira:wrm-data.sprint-goal-onboarding") || {};
    SprintView._sprintGoalOnboardingDialog = null;

    /**
     * Initializes sprint view listeners
     */
    SprintView.init = function () {
        var doc = AJS.$(document);

        // start sprint
        doc.on('click', '.js-sprint-start', SprintView.handleStartSprintClick);

        // Linked pages
        doc.on('click', '.js-sprint-container .js-view-entity-pages', SprintView.handleViewSprintLinkedPages);

        // Assigned work dialog
        doc.on('click', '.js-assigned-work-dialog-trigger', SprintView.handleViewAllAssignedWork);
    };

    SprintView._initSprintGoalOnboardingHandlers = _.once(function () {
        GH.RapidBoard.ViewController.on('before:navigate', function () {
            SprintView._hideSprintGoalOnboardingDialog();
        });
    });

    SprintView._hideSprintGoalOnboardingDialog = function () {
        if (SprintView._sprintGoalOnboardingDialog) {
            SprintView._sprintGoalOnboardingDialog.hide();
            SprintView._sprintGoalOnboardingDialog = null;
        }
    };

    /**
     * Event handler when 'Start Sprint' button is clicked.
     */
    SprintView.handleStartSprintClick = function (event) {
        if (AJS.$(event.target).hasClass('disabled')) {
            return;
        }
        var sprintId = SprintView.getSprintIdForElement(event.target);
        SprintController.openStartSprintDialog(sprintId);
    };

    /**
     * Event handler for when the 'view linked pages' link is clicked
     * @param event
     */
    SprintView.handleViewSprintLinkedPages = function (event) {
        var $trigger = AJS.$(event.target);
        var sprintId = SprintView.getSprintIdForElement($trigger);
        var canEdit = true; // TODO this needs to be retrieved from a model somewhere
        GH.LinkedPagesController.showDialogForSprint(GH.LinkedPagesController.PLAN, sprintId, $trigger);
    };

    /**
     * Event handler when 'Delete Sprint' button is clicked.
     */
    SprintView.handleDeleteSprintClick = function (event) {
        if (AJS.$(event.target).hasClass('disabled')) {
            return;
        }
        var sprintId = SprintView.getSprintIdForElement(event.target);
        SprintController.deleteSprint(sprintId);
    };

    SprintView.handleViewAllAssignedWork = function (event) {
        event.preventDefault();

        var sprintId = SprintView.getSprintIdForElement(event.target);
        var sprintModel = BacklogModel.getSprintModel(sprintId);

        // render bigger dialog for the additional tracking statistic column
        var widthClassValue = "medium";
        if (BacklogModel.trackingStatistic.isEnabled) {
            widthClassValue = "large";
        }

        var dialog = new AssignedWorkDialog({
            id: 'assigned-work-dialog',
            assignedWorkStats: BacklogStatistics.getAssignedWorkStats(sprintModel.getIssueList()),
            sprintName: sprintModel.getName(),
            estimateStatisticName: BacklogModel.estimationStatistic.name,
            trackingStatisticName: BacklogModel.trackingStatistic.name,
            widthClass: widthClassValue
        });
        dialog.show();

        // Hide tipsy so it doesn't continue to show under the dialog's blanket
        GH.Tooltip.hide('.js-assigned-work-dialog-trigger');

        new AnalyticsTracker('gh.assignedWork.dialog').trigger('show');
    };

    // Expander to reveal meta data (formerly named twix/twixie)
    SprintView.registerTwixies = function () {
        AJS.$('#ghx-backlog').undelegate('.ghx-backlog-container .ghx-expander', 'click.sprintTwix');
        AJS.$('#ghx-backlog').delegate('.ghx-backlog-container .ghx-expander', 'click.sprintTwix', SprintView.handleTwixieClick);
    };

    /**
     * Called when the user clicks on a sprint twixie
     */
    SprintView.handleTwixieClick = function (e) {
        var el = AJS.$(this).closest('.ghx-backlog-container');
        var sprintId = el.attr('data-sprint-id');
        SprintController.toggleTwixie(sprintId);
    };

    /**
     * Updates the twixie state for a given sprint
     */
    SprintView.updateTwixieState = function (sprintId) {
        var open = SprintController.isTwixieOpen(sprintId);
        var $container = AJS.$('.js-sprint-container[data-sprint-id="' + sprintId + '"]');
        var $header = $container.find('.ghx-backlog-header');
        var $wrapper = $header.parent('.sticky-wrapper');
        if (open) {
            $container.removeClass('ghx-closed').addClass('ghx-open');
        } else {
            $container.removeClass('ghx-open').addClass('ghx-closed');
            if ($header.hasClass('stuck')) {
                $wrapper.get(0).scrollIntoView(true);
            }
        }
        // Re-calculate offsets for sticky header and fix width for IE11
        GH.BacklogView.adjustStickyHeader();
    };

    //
    // Shared
    //

    SprintView.updateSprintViewForModel = function (sprintModel) {
        SprintView._updateSprint(sprintModel);
    };

    /**
     * Renders all open sprints
     * @param issueRenderData
     */
    SprintView.renderAllSprints = function (issueRenderData) {
        var sprintModels = BacklogModel.getSprintModels();

        var $sprintsContainer = $('#ghx-backlog').find('.ghx-sprint-group').empty();

        _.each(sprintModels, function (sprintModel, i) {
            // render the sprint container
            var $sprintContainer = AJS.$(GH.tpl.sprintview.renderSprintContainer({
                sprintData: sprintModel.getSprintData(),
                isMetaExpanded: SprintController.isTwixieOpen(sprintModel.getSprintId()),
                isKanbanColumn: GH.RapidBoard.State.isKanbanBoard()
            }));

            // render the sprint content
            var content = SprintView._renderSprintContent(sprintModel, issueRenderData);

            // add the content
            $sprintContainer.append(content);

            // add the rendered container to the dom
            $sprintsContainer.append($sprintContainer);

            if (i === 0 && SprintView.shouldDisplaySprintGoalOnboarding(sprintModel)) {
                SprintView._initSprintGoalOnboardingHandlers();
                SprintView.displaySprintGoalOnboarding(sprintModel.getSprintId());
            }
        });
        SprintView.enableTooltips();
    };

    /**
     * Decide if we should display sprint goal onboarding dialog next to a sprint
     * @param sprintModel model representing the sprint
     */
    SprintView.shouldDisplaySprintGoalOnboarding = function (sprintModel) {
        var isSprintGoalEnabled = FeatureFlagManager.isFeatureEnabled(DARK_FEATURE_SPRINT_GOAL_KEY);
        var hasSprintGoal = sprintModel.getSprintData() && sprintModel.getSprintData().goal;
        if (!isSprintGoalEnabled || SprintView.sprintGoalOnboardingState.isSprintGoalOnboardingClosed || hasSprintGoal) {
            return false;
        }
        return true;
    };

    /**
     * Displays inline dialog about sprint goal functionality
     * @param sprintId ID of the sprint next to which onboarding dialog should be shown
     */
    SprintView.displaySprintGoalOnboarding = function (sprintId) {
        if (!_.isFinite(sprintId) || sprintId < 0 || SprintView._sprintGoalOnboardingDialog) {
            return;
        }
        var $sprintsContainer = $('#ghx-backlog').find('.ghx-sprint-group');
        var $sprintContainer = $sprintsContainer.find('.js-sprint-container[data-sprint-id="' + sprintId + '"]');
        var $anchor = $sprintContainer.find('.js-sprint-actions-trigger');
        if (!$anchor.length) {
            return;
        }

        var helptip = SprintView._sprintGoalOnboardingDialog = new HelpTip({
            title: formatter.I18n.getText("gh.sprint.goal.onboarding.header"),
            bodyHtml: formatter.I18n.getText("gh.sprint.goal.onboarding.message"),
            anchor: $anchor,
            showCloseButton: true,
            closeButtonText: formatter.I18n.getText("gh.boards.kanplan.acknowledge.finish"),
            callbacks: {
                init: function init() {
                    helptip.view.$el.find('.helptip-close').on('click', function () {
                        return GH.Ajax.post({ url: '/xboard/plan/sprintGoalOnboarding/closeSprintGoalOnboarding' }).done(function () {
                            SprintView.sprintGoalOnboardingState.isSprintGoalOnboardingClosed = true;
                        });
                    });
                }
            }
        });

        $sprintsContainer.on('click', '.js-sprint-actions-trigger', function () {
            SprintView._hideSprintGoalOnboardingDialog();
        });

        helptip.show({ force: true });
    };

    /**
     * Redraws a sprint which has previously been drawn
     *
     * @param sprintModel the model of the sprint to redraw
     */
    SprintView._updateSprint = function (sprintModel) {
        GH.Logger.log("Redrawing sprint " + sprintModel.getSprintId(), GH.Logger.Contexts.ui);

        // find the right container
        var existingSprint = AJS.$('.js-sprint-container[data-sprint-id="' + sprintModel.getSprintId() + '"]');
        if (existingSprint.length === 0) {
            return;
        }

        // render the sprint content
        var issueRenderData = BacklogController.calculateIssueRenderData();
        var sprintContentHtml = SprintView._renderSprintContent(sprintModel, issueRenderData);

        // replace the current dom
        existingSprint.empty().append(sprintContentHtml);

        SprintView.enableTooltips();
    };

    /**
     * Renders the content of an active sprint, that is, the header and issue list
     */
    SprintView._renderSprintContent = function (sprintModel, issueRenderData) {
        var issueListStats = SprintView.calculateSprintIssueListStats(sprintModel, sprintModel.getIssueList());
        var sprintData = sprintModel.getSprintData();
        var manageSprintData = SprintView._getManageSprintData(sprintData);

        //These variables only make sense if we're rendering a kanban board. When rendering kanban board in plan mode,
        //we "model" the selected for dev column as a sprint object. When rendering the selected for dev column in plan
        //mode, we'd like to show the minimum and maximum issue count for the column.
        var minIssueCount = GH.RapidBoard.State.isKanbanBoard() ? sprintData.column.min : undefined;
        var maxIssueCount = GH.RapidBoard.State.isKanbanBoard() ? sprintData.column.max : undefined;

        var visibleRanks = sprintModel.getIssueList().getVisibleRankables();
        var issuesMap = sprintModel.getIssueList().getAllIssues();
        var issues = IssueListUtil.issueListToTree(visibleRanks.map(function (key) {
            return issuesMap[key];
        }), BacklogModel.getAllIssuesWithMissingParents());

        var isSprintGoalEnabled = FeatureFlagManager.isFeatureEnabled(DARK_FEATURE_SPRINT_GOAL_KEY);

        // show sprint actions (meatball menu) button:
        // - for sprints without goal IF sprint goals are enabled
        // - for all not active sprints
        var showSprintActions = GH.RapidBoard.State.isScrumBoard() && (isSprintGoalEnabled && !sprintData.goal || sprintData.state === 'FUTURE');

        // show start button:
        // - for all inactive sprints IF parallelSprints feature is enabled
        // - for top most inactive sprint IF only one active sprint is allowed
        var showSprintStartButton = GH.RapidBoard.State.isScrumBoard() && (manageSprintData.parallelSprints && sprintData.state === 'FUTURE' || manageSprintData.firstFutureSprintId === sprintData.id);

        IssueListUtil.applySubtaskCount(issues, sprintModel.getIssueList(), issueRenderData.hiddenIssues);
        SubtasksExpandingController.applyExpandedState(issues);

        var params = {
            sprintData: sprintData,
            manageSprintData: manageSprintData,
            isRankable: BacklogModel.isRankable(),
            showLinkedPages: BacklogModel.supportsPages() && GH.RapidBoard.State.isScrumBoard(),
            linkedPagesCount: sprintModel.getLinkedPagesCount(),
            issues: issues,
            issueListStats: issueListStats,
            epics: GH.EpicController.getEpicModel().getEpicList().getAllIssues(),
            issueRenderData: issueRenderData,
            statLabel: BacklogModel.estimationStatistic.name,
            trackLabel: BacklogModel.trackingStatistic.name,
            showEpic: GH.RapidBoard.State.getEpicShownOnRapidBoard(),
            showAssignedWorkStats: GH.RapidBoard.State.isScrumBoard(),
            showSprintActions: showSprintActions,
            showSprintStartButton: showSprintStartButton,
            showSprintFooter: GH.RapidBoard.State.isScrumBoard(),
            minIssueCount: minIssueCount,
            maxIssueCount: maxIssueCount,
            issueCount: Object.keys(issues).length,
            isKanbanBoard: GH.RapidBoard.State.isKanbanBoard(),
            isSprintGoalEnabled: isSprintGoalEnabled
        };

        return AJS.$(GH.tpl.sprintview.renderSprintContent(params));
    };

    /**
     * Updates the sprint statistics.
     * This function is invoked after filtering issues
     */
    SprintView.updateSprintStatistics = function () {
        var sprintModels = BacklogModel.getSprintModels();

        _.each(sprintModels, function (sprintModel) {
            // fetch the stats
            var issueListStats = SprintView.calculateSprintIssueListStats(sprintModel, sprintModel.getIssueList());

            SprintView._updateSprintStatisticsDom(sprintModel, issueListStats, false);
        });
    };

    /**
     * Updates the statistic for a dragged sprint
     * @param sprintModel the sprint model of the dragged sprint
     * @param issueList the list of issues currently part of the sprint, used to calculate statistics
     */
    SprintView.updateSprintStatisticForDraggedSprint = function (sprintModel, issueList) {
        // fetch the stats
        var issueListStats = SprintView.calculateSprintIssueListStats(sprintModel, issueList);

        SprintView._updateSprintStatisticsDom(sprintModel, issueListStats, true);
    };

    SprintView.markAsOvertaken = function (affectedModels) {
        var backlogAffected = false;
        var affectedSprintIds = [];
        _.each(affectedModels, function (model) {
            if (SprintModel.isSprintModel(model)) {
                affectedSprintIds.push(model.getSprintId());
            } else {
                backlogAffected = true;
            }
        });

        AJS.$('.js-sprint-container').each(function () {
            var $this = AJS.$(this);
            if (_.contains(affectedSprintIds, SprintView.getSprintIdForElement($this))) {
                $this.addClass('ghx-overtaken');
            } else {
                $this.removeClass('ghx-overtaken');
            }
        });
        AJS.$('.ghx-everything-else').each(function () {
            var $this = AJS.$(this);
            if (backlogAffected) {
                $this.addClass('ghx-overtaken');
            } else {
                $this.removeClass('ghx-overtaken');
            }
        });
    };

    SprintView._updateSprintStatisticsDom = function (sprintModel, issueListStats, updatedDraggedMarker) {

        // find the dom element to replace content in
        var $sprint = AJS.$('.js-sprint-container[data-sprint-id="' + sprintModel.getSprintId() + '"]');

        // re-render the issue count info
        var activeSprintIssueCountHtml = SprintView._renderSprintIssueCount(issueListStats);
        $sprint.find('.ghx-issue-count').replaceWith(activeSprintIssueCountHtml);

        if (GH.RapidBoard.State.isScrumBoard()) {
            // re-render assignee avatars
            $sprint.find('.ghx-assigned-work-stats').replaceWith(GH.tpl.sprintview.renderAssignedWork({
                assignedWork: issueListStats.assignedWorkStats.assigned,
                estimateStatisticName: BacklogModel.estimationStatistic.name,
                trackingStatisticName: BacklogModel.trackingStatistic.name
            }));
        }

        // also update the sprint footer if this is a future sprint
        if (sprintModel.isFutureSprint()) {
            var statsHtml = SprintView._renderFutureSprintFooterContent(issueListStats);
            $sprint.find('.js-marker-divider').empty().append(statsHtml);

            if (updatedDraggedMarker) {
                AJS.$('.js-marker-divider[data-sprint-id="' + sprintModel.getSprintId() + '"]').empty().append(statsHtml);
            }
        }

        SprintView.enableTooltips();
    };

    /**
     * Renders the meta information of a sprint (start, end date)
     */
    SprintView._renderSprintMetaInfos = function (sprintModel) {
        return GH.tpl.sprintview.renderSprintMetaInfos({
            sprintData: sprintModel.getSprintData()
        });
    };

    /**
     * Re-renders the active sprint issue count section
     */
    SprintView._renderSprintIssueCount = function (issueListStats) {
        // render
        return GH.tpl.planissuelist.renderIssueCount({
            issueListStats: issueListStats
        });
    };

    /**
     * Re-renders the active sprint issue count section
     */
    SprintView._renderFutureSprintFooterContent = function (issueListStats) {
        if (GH.RapidBoard.State.isScrumBoard()) {
            return GH.tpl.sprintview.renderFutureSprintFooterContent({
                issueListStats: issueListStats
            });
        }

        return "";
    };

    /**
     * Returns sprint management specific data
     * @return {{hasOpenSprints: boolean, canManageSprints: boolean, parallelSprints: *, firstFutureSprintId: (*|Function|id|Function|SPAN.id|id|SPAN.id|id|Function|Function|id|Function|id|Function|id|id|id|id|id|id|id|Selector.xpath.id|Selector.criteria.id|Function|Function|id|id|._newInst.id|id|*|id|GH.RapidBoard.Config.FilterModel.id|Link.id|string|GH.RapidViewConfig.currentData.id|number|string|string|string|string|id|string|string|id|id|.this.attributes.id|id|.this.attributes.id|id|tabberObj.id|id|X.id|X.id|X.id|X.id|id|string|X.id|X.id|X.id|columnData.id|id|Z.id|Z.id|Z.id|Z.id|Z.id|Z.id|Z.id|string|Z.id|Z.id|Z.id|Z.id|Z.id|Z.id|String|GH.RestfulTable.id|id|string|id|.this.attributes.id|newState.id|id|$.id|$.id|$.id|$.id|$.id|$.id|$.id|id|$.id|$.id|$.id|$.id|$.id|id|$.id|$.id|id|id|$.id|id|$.id|String|Ext.Element.id|replaceWith.id|id|this.el.id|Ext.Element.id|replaceWith.id|id|this.el.id|Ext.Element.id|replaceWith.id|id|this.el.id|D.id|z.id|id|i._drag.id|id|core.id|Selector.root.id|string|id|Page.id|Marker.id|Board.id|Issue.id|id|uniqueId.id|id|String|replaceWith.id|id|this.el.id)}}
     * @private
     */
    SprintView._getManageSprintData = function (sprintData) {
        var activeSprints = BacklogModel.getActiveSprintModels();
        var futureSprints = BacklogModel.getFutureSprintModels();
        var hasOpenSprints = activeSprints.length > 0;
        var parallelSprints = GH.RapidBoard.parallelSprints;
        var firstFutureSprintId = futureSprints.length > 0 ? futureSprints[0].getSprintData().id : null;
        var canStartSprint = (parallelSprints || !hasOpenSprints) && sprintData.canUpdateSprint;
        var canManageSprints = BacklogModel.canManageSprints();

        return {
            canStartSprint: canStartSprint,
            canUpdateSprint: sprintData.canUpdateSprint,
            canManageSprints: canManageSprints,
            hasOpenSprints: hasOpenSprints,
            parallelSprints: parallelSprints,
            firstFutureSprintId: firstFutureSprintId
        };
    };

    /**
     * Calculates the issue list stats for a sprint
     */
    SprintView.calculateSprintIssueListStats = function (sprintModel, issueList) {
        var issueListStats = BacklogStatistics.getIssueListStats(issueList, true, sprintModel.isActiveSprint());
        // future sprints also need sprintStats (shown on the marker)
        if (sprintModel.isFutureSprint()) {
            var sprintStats = BacklogStatistics.getIssueListEstimateStatistics(issueList);
            _.extend(issueListStats, sprintStats);
        }

        // active sprints require progress indicators
        else {
                issueListStats.progressIndicators = BacklogStatistics.calculateSprintProgress(issueList, BacklogController.rapidViewConfig.columns, BacklogModel.estimationStatistic);
            }
        issueListStats.assignedWorkStats = BacklogStatistics.getAssignedWorkStats(issueList);

        if (GH.RapidBoard.State.isKanbanBoard()) {
            issueListStats = _.extend({}, issueListStats, BacklogStatistics.calculateStatisticsFieldValue(issueList, BacklogController.rapidViewConfig.statisticConfig));
        }
        return issueListStats;
    };

    // Utilities

    /**
     * Returns the sprint id of the sprint container for an element.
     * @param {HTMLElement} element
     * @returns {number|null}
     */
    SprintView.getSprintIdForElement = function (element) {
        var sprintElem = AJS.$(element).closest('.js-sprint-container');
        if (!sprintElem.length) {
            return null;
        }
        return parseInt(sprintElem.attr('data-sprint-id'), 10);
    };

    SprintView.getSprintIdForMarker = function (marker) {
        var markerElem = AJS.$(marker).closest('.js-marker');
        if (!markerElem.length) {
            return null;
        }
        var sprintId = parseInt(markerElem.attr('data-sprint-id'), 10);
        return sprintId;
    };
    /**
     * Tooltips for a sprint
     */
    SprintView.enableTooltips = function () {
        // Enable tipsy with default behaviour for the sprint start buttons,
        // since we want to take the advantage of default delayOut
        // interval so that user can click on the link within the tipsy.
        $('.js-sprint-start').tooltip({
            html: true,
            title: 'data-tooltip'
        });

        $('.ghx-sprint-goal .js-edit-sprintGoal-trigger').tooltip({
            title: 'data-fieldvalue'
        });

        // tipsy for the assigned work summary button
        GH.Tooltip.tipsify({
            selector: '.js-assigned-work-dialog-trigger'
        });
    };

    return SprintView;
});