define('jira-agile/rapid/ui/plan/subtasks-expanding-controller', ['underscore', 'require', 'jira/analytics'], function (_, require, analytics) {
    var SubtasksExpandingController = {};

    var CSS_CLASS_EXPANDED = 'ghx-subtasks-expanded';
    var COLLAPSED_PARENTS_KEY = 'collapsed-parents';

    var GlobalEvents = require('jira-agile/rapid/global-events');
    var PlanIssueListView = require('jira-agile/rapid/ui/plan/plan-issue-list-view');
    var AnalyticsHelper = require('jira-agile/rapid/ui/plan/analytics-helper');
    var BacklogModel;

    // Resolve circular dependency
    GlobalEvents.on("pre-initialization", function () {
        SubtasksExpandingController._preInitialization();
    });

    SubtasksExpandingController._preInitialization = function () {
        BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    };

    /**
     * Storing data
     */

    /**
     * Set collapsed state of subtasks group
     * @param {Number} issueKey
     * @param {Boolean} isExpanded
     */
    function setParentExpandedState(issueKey, isExpanded) {
        if (!isExpanded) {
            SubtasksExpandingController._collapsedParents[issueKey] = true;
        } else {
            delete SubtasksExpandingController._collapsedParents[issueKey];
        }

        GH.RapidBoard.State.setBoardSetting(COLLAPSED_PARENTS_KEY, SubtasksExpandingController._collapsedParents);
    }

    /**
     * Get all collapsed parent tasks.
     * @returns {Array}
     */
    function getCollapsedParentsKeys() {
        return GH.RapidBoard.State.getBoardSetting(null, COLLAPSED_PARENTS_KEY, {});
    }

    /**
     * Subtasks expanding controller definition
     */

    SubtasksExpandingController._collapsedParents = getCollapsedParentsKeys();

    SubtasksExpandingController.applyExpandedState = function (issues) {
        var issueArray = _.values(issues);

        issueArray.forEach(function (issue) {
            issue.expanded = !SubtasksExpandingController.isIssueCollapsed(issue.key);
        });
    };

    /**
     * @param {jQuery} $issueElement - DOM reference
     * @param {boolean} expandedState - expanded state (optional)
     * @returns {boolean}
     */
    SubtasksExpandingController.toggleExpandState = function ($issueElement, expandedState) {
        var $parentContainer = PlanIssueListView.getClosestParent($issueElement);
        var issueKey = PlanIssueListView.getIssueKey($parentContainer);

        if (!_.isUndefined(expandedState)) {
            $parentContainer.toggleClass(CSS_CLASS_EXPANDED, !!expandedState);
        } else {
            $parentContainer.toggleClass(CSS_CLASS_EXPANDED);
        }

        var isExpanded = $parentContainer.hasClass(CSS_CLASS_EXPANDED);

        //remember expand state
        setParentExpandedState(issueKey, isExpanded);

        return isExpanded;
    };

    /**
     * @param {String} issueKey key of the issue
     * @param {boolean} expandedState desired expanded state (optional)
     * @returns {boolean}
     */
    SubtasksExpandingController.toggleExpandStateForKey = function (issueKey, expandedState) {
        var $issueElement = PlanIssueListView.getIssueByKey(issueKey);

        return SubtasksExpandingController.toggleExpandState($issueElement, expandedState);
    };

    /**
     * Get all subtasks that are collapsed in from given parents list
     * @param {Array} selectedIssueKeys list of issues to check
     * @returns {Array} list of all collapsed subtasks
     */
    SubtasksExpandingController.getCollapsedSubTasksToSelection = function (selectedIssueKeys) {
        var collapsedChildKeys = [];

        selectedIssueKeys.forEach(function (issueKey) {
            if (!SubtasksExpandingController.isIssueCollapsed(issueKey)) {
                return;
            }

            var model = BacklogModel.findModelWithIssue(issueKey);
            if (!model) {
                return;
            }

            var subtasks = model.issueList.getSubtasks(issueKey).map(function (issue) {
                return issue.key;
            });
            collapsedChildKeys = collapsedChildKeys.concat(subtasks);
        });

        return collapsedChildKeys;
    };

    /**
     * Check if given issue is collapsed
     * @param issueKey issue key to check
     * @returns {boolean} true if issue is collapsed, false otherwise
     */
    SubtasksExpandingController.isIssueCollapsed = function (issueKey) {
        return !!SubtasksExpandingController._collapsedParents[issueKey];
    };

    /**
     * Send analytics event - someone expanded or collapsed subtasks group
     * @param isExpanded
     */
    SubtasksExpandingController.sendAnalytics = function (isExpanded) {
        analytics.send({
            name: 'jira-software.plan.subtasks.expanded.changed',
            properties: _.extend({
                isExpanded: isExpanded
            }, AnalyticsHelper.baseEventData())
        });
    };

    return SubtasksExpandingController;
});