GH.StartWizardView = {};

GH.StartWizardView.dialog = undefined;
GH.StartWizardView.model = undefined;
GH.StartWizardView.stepsNameStack = undefined;
GH.StartWizardView.pages = {};
GH.StartWizardView.filterPicker = undefined;
GH.StartWizardView.projectPicker = undefined;

(function () {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GH.StartWizardView.analytics = new AnalyticsTracker("gh.create.wizard");
})();

// These callbacks provide integration points for the create project wizard, allowing other plugins to add additional
// options and functionality to the dialog.
//
// This was added to support the cross-product project creation functionality.
// See: https://bitbucket.org/atlassian/atlassian-project-creation
//      https://extranet.atlassian.com/display/N/2013/07/25/Project+create+update
//
GH.StartWizardView.wizardStepPreRenderCallbacks = [];
GH.StartWizardView.wizardStepPostRenderCallbacks = [];
GH.StartWizardView.wizardStepOnValidateCallbacks = [];
GH.StartWizardView.wizardStepOnCreateCallbacks = [];

GH.StartWizardView.steps = {
    methodologySelection: {
        render: function render(model) {
            var params = {
                isSampleDataAvailable: GH.Config.isSampleDataAvailable,
                canCreateProject: GH.UserData.canCreateProject()
            };
            return {
                header: AJS.I18n.getText("gh.wizard.heading.methodology"),
                content: GH.tpl.startwizard.renderMethodologySelectionStep(params)
            };
        },
        bind: function bind($content, model) {
            AJS.$("#ghx-wizard-methodology-scrum").click(function (e) {
                e.preventDefault();
                model.sampleData = false;
                GH.StartWizardView.selectMethodology("scrum", model);
            });
            AJS.$("#ghx-wizard-methodology-kanban").click(function (e) {
                e.preventDefault();
                model.sampleData = false;
                GH.StartWizardView.selectMethodology("kanban", model);
            });
            AJS.$("#ghx-wizard-methodology-scrum-sample").click(function (e) {
                e.preventDefault();
                model.sampleData = true;
                GH.StartWizardView.selectMethodology("scrum", model);
            });
            AJS.$("#ghx-wizard-methodology-kanban-sample").click(function (e) {
                e.preventDefault();
                model.sampleData = true;
                GH.StartWizardView.selectMethodology("kanban", model);
            });
        }
    },
    methodSelection: {
        render: function render(model) {
            var selectedOption = model.method;
            if (!selectedOption) {
                selectedOption = GH.UserData.userConfig.canCreateProject ? "newProject" : "existingProject";
            }
            return {
                header: AJS.I18n.getText("gh.wizard.heading.method"),
                content: GH.tpl.startwizard.renderMethodSelectionStep(_.extend({}, model, {
                    canCreateProject: GH.UserData.userConfig.canCreateProject,
                    hasProjectsAccessible: GH.UserData.userConfig.hasProjectsAccessible,
                    hasFiltersAccessible: GH.UserData.userConfig.hasFiltersAccessible,
                    selectedOption: selectedOption
                }))
            };
        },
        bind: function bind($content) {},
        next: function next($content, model) {
            // redirect to the next step according to the selected board creation method
            var selectedValue = $content.find('input[name=ghx-wizard-method]').filter(':checked').val();

            // store the selected method in the model to restore it in case we need to go back to this step
            model.method = selectedValue;

            var deferred = AJS.$.Deferred();
            if (selectedValue === 'newProject') {
                deferred.resolve('projectCreation');
            } else if (selectedValue === 'existingProject') {
                deferred.resolve('projectSelection');
            } else if (selectedValue === 'existingFilter') {
                deferred.resolve('filterSelection');
            } else {
                deferred.reject(new Error('Unknown board creation method : ' + selectedValue)); // @todo i18n?
            }

            return deferred.promise();
        }
    },
    projectCreation: {
        render: function render(model) {
            model.prefix = "project";
            return GH.StartWizardView.renderProjectCreation({
                user: GH.UserData.userConfig,
                prefix: "project",
                sampleInfo: {
                    projectName: "",
                    projectKey: ""
                }
            });
        },
        bind: function bind($content, model) {
            GH.StartWizardView.bindProjectCreation($content, model);
        },
        completeActionName: AJS.I18n.getText("gh.wizard.finish"),
        complete: function complete(model) {
            return GH.StartWizardView.createProject(GH.StartWizardView.dialog, model);
        }
    },
    sampleProjectCreation: {
        render: function render(model) {
            model.prefix = "sample";
            return GH.StartWizardView.renderProjectCreation({
                user: GH.UserData.userConfig,
                prefix: "sample",
                sampleInfo: model.sampleInfo
            });
        },
        bind: function bind($content, model) {
            GH.StartWizardView.bindProjectCreation($content, model);
        },
        completeActionName: AJS.I18n.getText("gh.wizard.finish"),
        complete: function complete(model) {
            return GH.StartWizardView.createSampleProject(GH.StartWizardView.dialog, model);
        }
    },
    projectSelection: {
        render: function render(model) {
            return {
                header: AJS.I18n.getText("gh.wizard.heading.project.existing"),
                content: GH.tpl.startwizard.renderProjectSelectionStep()
            };
        },
        bind: function bind($content, model) {
            GH.StartWizardView.bindProjectSelection($content, model);
        },
        completeActionName: AJS.I18n.getText("gh.wizard.finish"),
        complete: function complete(model) {
            return GH.StartWizardView.validateProjectSelection(model);
        }
    },
    filterSelection: {
        render: function render(model) {
            return {
                header: AJS.I18n.getText("gh.wizard.heading.filter"),
                content: GH.tpl.startwizard.renderFilterSelectionStep()
            };
        },
        bind: function bind($content, model) {
            GH.StartWizardView.bindFilterSelection($content, model);
        },
        completeActionName: AJS.I18n.getText("gh.wizard.finish"),
        complete: function complete(model) {
            return GH.StartWizardView.validateFilterSelection(model);
        }
    }
};

GH.StartWizardView.show = function (options) {
    if (!GH.StartWizardView.dialog) {
        GH.StartWizardView.dialog = GH.Dialog.create({
            width: 840,
            height: 400,
            id: 'ghx-wizard-dialog',
            onCancelFn: function onCancelFn() {
                GH.StartWizardView.disposeWizard();
                GH.StartWizardView.analytics.trigger("cancel");
            }
        });

        GH.StartWizardView.model = {};
        GH.StartWizardView.stepsNameStack = [];
        GH.StartWizardView.dialog.show();
        GH.StartWizardView.analytics.trigger("start");
        GH.StartWizardView.preventSubmitOnEnter();
    }
};

GH.StartWizardView.preventSubmitOnEnter = function () {
    var preventSubmitOnEnter = function preventSubmitOnEnter(event) {
        // allow enter for filter chooser
        if (event.keyCode == 13 && event.target.id != "ghx-wizard-filter-select-field") {
            event.preventDefault();
            return false;
        }
    };
    AJS.$("input").keydown(preventSubmitOnEnter);
};

GH.StartWizardView.showStep = function (stepName) {
    if (!_.has(GH.StartWizardView.steps, stepName)) {
        throw new Error('Unknown wizard step ' + stepName); // @todo i18n?
    }

    // save local reference
    var dialog = GH.StartWizardView.dialog;

    // seems that the safest way to change the content of the dialog is to add a new page
    if (!GH.StartWizardView.pages[stepName]) {

        var stepDescriptor = GH.StartWizardView.steps[stepName];
        var renderResult = stepDescriptor.render(GH.StartWizardView.model);

        _.each(GH.StartWizardView.wizardStepPreRenderCallbacks, function (callback) {
            callback(stepName);
        });

        dialog.addPage();
        GH.StartWizardView.pages[stepName] = dialog.curpage;
        dialog.addHeader(renderResult.header);

        // live (jQuery) content
        var $content = AJS.$(renderResult.content);

        // panel title is not used since we use dialog header for that
        dialog.addPanel(null, $content, 'ghx-wizard-panel');

        _.each(GH.StartWizardView.wizardStepPostRenderCallbacks, function (callback) {
            callback(stepName, dialog);
        });

        // back button makes sense only if the steps stack is not empty
        if (GH.StartWizardView.stepsNameStack.length > 0) {
            dialog.addButton('Back', function () {
                var previousStep = GH.StartWizardView.stepsNameStack.pop();
                // go back to previous step
                GH.StartWizardView.showStep(previousStep);
            }, "aui-button");
        }

        // only add complete button if the step has a complete function
        if (_.has(stepDescriptor, 'complete')) {
            var goComplete = function goComplete() {
                var btn = AJS.$(this);
                var disabled = btn.attr("aria-disabled");
                if (disabled == "true") {
                    return;
                }
                stepDescriptor.complete(GH.StartWizardView.model);
            };
            dialog.addButton(stepDescriptor.completeActionName, goComplete, "aui-button aui-button-primary js-wizard-button-complete");
        }
        // only add next button if the step has a next function
        else if (_.has(stepDescriptor, 'next')) {
                var goNext = function goNext() {
                    var btn = AJS.$(this);
                    var disabled = btn.attr("aria-disabled");
                    if (disabled == "true") {
                        return;
                    }

                    var nextStepPromise = stepDescriptor.next($content, GH.StartWizardView.model);

                    // the next method return value is a promise that is resolved with the next step to jump to or rejected with
                    // an error if something went wrong
                    var onOk = function onOk(nextStep) {
                        GH.StartWizardView.stepsNameStack.push(stepName);
                        GH.StartWizardView.showStep(nextStep);
                    };
                    var onFail = function onFail(cause) {
                        // todo something more clever here
                        console.log(cause);
                        dialog.updateHeight();
                    };
                    var validateCallbackSuccess = true;
                    _.each(GH.StartWizardView.wizardStepOnValidateCallbacks, function (callback) {
                        if (!callback(stepName, dialog)) {
                            validateCallbackSuccess = false;
                        }
                    });
                    nextStepPromise.done(function (nextStep) {
                        if (validateCallbackSuccess) {
                            onOk(nextStep);
                        } else {
                            onFail("GH project create extension callback returned failure");
                        }
                    }).fail(onFail);
                };
                dialog.addButton(AJS.I18n.getText("gh.rapid.common.next"), goNext, "aui-button aui-button-primary js-wizard-button-next");
            }

        dialog.addCancel(AJS.I18n.getText("gh.rapid.operations.cancel"), function () {
            dialog.cancel();
        });

        // attach event handler
        stepDescriptor.bind($content, GH.StartWizardView.model);
    } else {
        // known limitation - when revisiting a later step in the wizard, but an early choice changes the content of the
        // later step, this is currently not handled. see renderResult above.
        dialog.gotoPage(GH.StartWizardView.pages[stepName]);

        // removing any previous errors that were being displayed on the step
        GH.StartWizardView.dialog.popup.element.find(".dialog-panel-body .aui-message-error").remove();
    }

    dialog.updateHeight();
    GH.StartWizardView.preventSubmitOnEnter();
};

GH.StartWizardView.startWizard = function () {
    GH.StartWizardView.show();
    GH.StartWizardView.showStep('methodologySelection');
};

GH.StartWizardView.disposeWizard = function () {
    GH.StartWizardView.dialog = undefined;
    GH.StartWizardView.pages = {};
};

GH.StartWizardView.genStartFunction = function (type, sample) {
    return function () {
        GH.StartWizardView.show();
        GH.StartWizardView.model.sampleData = sample;
        GH.StartWizardView.selectMethodology(type, GH.StartWizardView.model);
    };
};

GH.StartWizardView.bindWelcomePage = function () {
    GH.Dialog.registerDialog(".js-wizard-scrum", GH.StartWizardView.genStartFunction("scrum", false));
    GH.Dialog.registerDialog(".js-wizard-scrum-sample", GH.StartWizardView.genStartFunction("scrum", true));
    GH.Dialog.registerDialog(".js-wizard-kanban", GH.StartWizardView.genStartFunction("kanban", false));
    GH.Dialog.registerDialog(".js-wizard-kanban-sample", GH.StartWizardView.genStartFunction("kanban", true));
};

/**
 * Transform the internal data into an object that gets serialised into JSON and then
 * deserialised into  com.atlassian.greenhopper.web.rapid.welcome.WelcomeCreateProjectRequest
 */
GH.StartWizardView.getCreateProjectRequest = function (model) {

    var project = model.project;

    var projectRequest = {
        projectName: project.name,
        projectKey: project.key,
        projectLeadUserName: project.lead,
        boardName: model.boardName,
        preset: model.methodology,
        sampleData: model.sampleData
    };

    return projectRequest;
};

/**
 * Take the data that we've collected through the wizard and send it to the server
 * Handles validation errors
 */
GH.StartWizardView.createProject = function (dialog, model) {
    GH.StartWizardView.populateProjectCreationProperties(model);

    var data = GH.StartWizardView.getCreateProjectRequest(model);

    dialog.disableControls();
    dialog.showSpinner();

    var settings = {
        url: '/welcome/createProject',
        data: data,
        errorContextMap: {
            'projectName': '#ghx-wizard-' + model.prefix + '-projectname',
            'projectKey': '#ghx-wizard-' + model.prefix + '-projectkey',
            'projectLead': '#ghx-wizard-' + model.prefix + '-projectlead'
        },
        timeout: 2 * 60 * 1000,
        deferErrorHandling: true
    };

    return GH.Ajax.put(settings).done(function (result) {
        result.success.sample = model.sampleData;
        GH.PersistentMessages.WelcomeMessage.show(result.success);

        // execute any callbacks and forward user to newly created rapid board if there are none
        var goToNextPage = function goToNextPage() {
            GH.StartWizardView.gotoCreatedView(result.success.rapidView.id, model.methodology);
        };

        function executeNextCallback() {
            if (GH.StartWizardView.wizardStepOnCreateCallbacks.length == 0) {
                goToNextPage();
            } else {
                var callback = GH.StartWizardView.wizardStepOnCreateCallbacks.pop();
                callback(dialog, executeNextCallback, model);
            }
        };

        executeNextCallback();
    }).fail(function (xhr, statusText, errorThrown, smartAjaxResult) {
        // handle soft timeout(request aborted by jQuery) message
        _.each(xhr.getGlobalErrors(), function (error) {
            if (error.status == 504) {
                error.message = GH.tpl.startwizard.createProjectTimeout({ manageBoardURL: GH.RapidBoard.getManageViewsPageUrl() });
                error.noAutoescape = true;
            }
        });
        GH.Ajax.handleContextualErrors(xhr, settings.errorContextMap);
        GH.Ajax.handleGlobalErrors(xhr, settings.errorContextMap);

        dialog.hideSpinner();
        dialog.enableControls();
        dialog.updateHeight();
    });
};

GH.StartWizardView.validateProjectSelection = function (model) {

    GH.StartWizardView.dialog.disableControls();
    GH.StartWizardView.dialog.showSpinner();

    var $boardName = AJS.$('#ghx-wizard-project-board-name');
    GH.Validation.clearContextualErrors(GH.StartWizardView.projectPicker.getElement());
    GH.Validation.clearContextualErrors($boardName);

    return GH.Ajax.post({
        url: '/rapidview/create/presets',
        data: {
            'name': $boardName.val(),
            'projectIds': GH.StartWizardView.projectPicker.getValue(),
            'preset': model.methodology
        },
        errorContextMap: {
            name: '#ghx-wizard-project-board-name',
            projectIds: GH.StartWizardView.projectPicker.selector
        }
    }).done(function (result) {
        console.log("mode: " + model);
        GH.StartWizardView.handleCreate(result, "simple", model.methodology);
    }).fail(function () {
        GH.StartWizardView.dialog.enableControls();
        GH.StartWizardView.dialog.hideSpinner();
        GH.StartWizardView.dialog.updateHeight();
    });
};

GH.StartWizardView.validateFilterSelection = function (model) {
    var $boardName = AJS.$('#ghx-wizard-filter-view-name');
    var $filterElement = GH.StartWizardView.filterPicker.getElement();
    var $filterSelect = AJS.$(GH.StartWizardView.filterPicker.selector);

    GH.Validation.clearContextualErrors($boardName);
    GH.Validation.clearContextualErrors($filterElement);

    var filterId = $filterSelect.val() ? $filterSelect.val()[0] : null;

    GH.StartWizardView.dialog.disableControls();
    GH.StartWizardView.dialog.showSpinner();

    return GH.Ajax.post({
        url: '/rapidview/create/advanced',
        data: {
            'name': $boardName.val(),
            'filterId': filterId,
            'methodology': model.methodology
        },
        errorContextMap: {
            name: '#ghx-wizard-filter-view-name',
            filterId: GH.StartWizardView.filterPicker.selector
        }
    }).done(function (result) {
        GH.StartWizardView.handleCreate(result, "simple", model.methodology);
    }).fail(function () {
        GH.StartWizardView.dialog.hideSpinner();
        GH.StartWizardView.dialog.enableControls();
        GH.StartWizardView.dialog.updateHeight();
    });
};

/**
 * Fire off some analytics, set up the message and then redirect somewhere.
 */
GH.StartWizardView.handleCreate = function (result, type, methodology) {
    var subType = type != methodology ? methodology : false;
    GH.PersistentMessages.CreateViewMessage.setView(result.success.id, result.success.name, type, subType);

    GH.StartWizardView.analytics.trigger("complete");
    GH.StartWizardView.gotoCreatedView(result.success.id, methodology);
};

GH.StartWizardView.gotoCreatedView = function (rapidViewId, methodology) {
    var targetView = methodology == "scrum" ? "planning" : null;
    GH.Navigation.gotoDefaultRapidBoardPage(rapidViewId, targetView);
};

GH.StartWizardView.selectMethodology = function (methodology, model) {
    model.methodology = methodology;

    GH.StartWizardView.stepsNameStack.push("methodologySelection");

    if (model.sampleData) {
        //        GH.StartWizardView.showStep('sampleProjectCreation');
        GH.Ajax.get({
            url: '/welcome/sampledataproject?preset=' + model.methodology
        }, "sampledataproject").done(function (result) {
            model.sampleInfo = result;
            GH.StartWizardView.showStep('sampleProjectCreation');
        }).fail(function () {
            GH.StartWizardView.showStep('sampleProjectCreation');
        });
    } else {
        var canCreateProject = GH.UserData.userConfig.canCreateProject;
        var hasProjectsAccessible = GH.UserData.userConfig.hasProjectsAccessible;
        var hasFiltersAccessible = GH.UserData.userConfig.hasFiltersAccessible;

        if (!canCreateProject && !hasFiltersAccessible) {
            // can't create a new project, no filters, must select an existing one
            GH.StartWizardView.showStep('projectSelection');
        } else if (!hasProjectsAccessible && !hasFiltersAccessible) {
            // no projects or filters, must create new project
            GH.StartWizardView.showStep('projectCreation');
        } else {
            GH.StartWizardView.showStep('methodSelection');
        }
    }
};

GH.StartWizardView.populateProjectCreationProperties = function (model) {
    // validate new project properties
    var $projectName = AJS.$('#ghx-wizard-' + model.prefix + '-projectname');
    var $projectKey = AJS.$('#ghx-wizard-' + model.prefix + '-projectkey');
    var $lead = AJS.$('#ghx-wizard-' + model.prefix + '-projectlead');

    GH.Validation.clearContextualErrors($projectName);
    GH.Validation.clearContextualErrors($projectKey);
    GH.Validation.clearContextualErrors($lead);

    model.project = {
        name: $projectName.val(),
        key: $projectKey.val().toUpperCase(),
        lead: AJS.$('#ghx-wizard-' + model.prefix + '-projectlead option:selected').val()
    };
};

GH.StartWizardView.createSampleProject = function (dialog, model) {
    model.sampleData = true;
    GH.StartWizardView.createProject(dialog, model);
};

GH.StartWizardView.validateBoardName = function ($boardName, model) {
    GH.Validation.clearContextualErrors($boardName);
    model.boardName = $boardName.val();

    if (!model.boardName) {
        GH.Validation.showContextualErrors($boardName, AJS.I18n.getText("gh.rapid.welcome.board.name.required"));
        return false;
    }
    return true;
};

GH.StartWizardView.renderProjectCreation = function (model) {
    return {
        header: AJS.I18n.getText("gh.wizard.heading.project.new"),
        content: GH.tpl.startwizard.renderProjectCreationStep(model)
    };
};

GH.StartWizardView.bindProjectSelection = function ($content, model) {
    GH.StartWizardView.projectPicker = new GH.ProjectPicker({
        selector: '#ghx-wizard-choose-project',
        change: function change() {
            GH.StartWizardView.dialog.updateHeight();
        },
        currentProjectId: JIRA.API && JIRA.API.Projects ? JIRA.API.Projects.getCurrentProjectId() : null
    });

    GH.StartWizardView.projectPicker.multiselect.keys.Return = function (e) {
        e.preventDefault();
    };

    GH.StartWizardView.attachDependantButton({
        buttonIndex: 1,
        selectors: ['#ghx-wizard-project-board-name', '#ghx-wizard-choose-project']
    });
};

GH.StartWizardView.bindProjectCreation = function ($content, model) {
    AJS.$('#ghx-wizard-' + model.prefix + '-projectkey').generateFrom(AJS.$('#ghx-wizard-' + model.prefix + '-projectname'), {
        maxNameLength: 64,
        maxKeyLength: 64
    });

    // this causes the user picker frother single select to be created
    JIRA.trigger(JIRA.Events.NEW_CONTENT_ADDED, [$content]);
    $content.trigger("contentRefresh");

    var dialog = GH.StartWizardView.dialog;
    var page = dialog.page[dialog.curpage];
    var button = page.button[1].item;

    GH.StartWizardView.attachDependantButton({
        buttonIndex: 1,
        selectors: ['#ghx-wizard-' + model.prefix + '-projectname', '#ghx-wizard-' + model.prefix + '-projectkey', '#ghx-wizard-' + model.prefix + '-projectlead']
    });
};

GH.StartWizardView.bindFilterSelection = function ($content, model) {
    /**
     * Update the display of the JIRA filter permissions once the filter frother selection changes. If there is no filter selected,
     * the view is cleared.
     */
    var updatePermissionInfo = function updatePermissionInfo(event, item) {
        var html = "";
        delete model.savedFilter;
        if (item && item.properties) {
            var savedFilter = item.properties.savedFilter;
            html = savedFilter != null ? GH.tpl.startwizard.renderPermissionInfo({ savedFilter: savedFilter }) : '';
            model.savedFilter = savedFilter;
        }
        AJS.$('#ghx-create-permissioninfo-container').html(html);
        GH.StartWizardView.dialog.updateHeight();
    };

    GH.StartWizardView.filterPicker = new GH.SavedFilterPicker({
        selector: '#ghx-wizard-filter-select',
        selected: updatePermissionInfo,
        stayActivated: true
    });

    GH.StartWizardView.filterPicker.show(model.savedFilter);
    GH.StartWizardView.filterPicker.singleSelect.keys.Return = function (e) {
        e.preventDefault();
    };

    GH.StartWizardView.attachDependantButton({
        buttonIndex: 1,
        selectors: ['#ghx-wizard-filter-view-name', '#ghx-wizard-filter-select']
    });
};

GH.StartWizardView.attachDependantButton = function (options) {
    var dialog = GH.StartWizardView.dialog;
    var page = dialog.page[dialog.curpage];
    var button = page.button[options.buttonIndex].item;

    GH.StartWizardView.dependantValidationListener({
        selectors: options.selectors,
        full: function full() {
            button.attr("aria-disabled", "false");
        },
        empty: function empty() {
            button.attr("aria-disabled", "true");
        }
    });
};

GH.StartWizardView.dependantValidationListener = function (options) {
    if (_.isEmpty(options.selectors)) {
        return;
    }

    options.full = options.full || AJS.$.noop();
    options.empty = options.empty || AJS.$.noop();
    options.nowait = options.nowait || false;

    function valueChanged() {
        var emptyFields = _.filter(options.selectors, function (sel) {
            var element = AJS.$(sel);
            return !AJS.$.trim(element.val());
        });

        if (_.isEmpty(emptyFields)) {
            return options.full();
        } else {
            return options.empty();
        }
    }

    function callValueChanged() {
        if (options.nowait) {
            valueChanged();
        } else {
            setTimeout(valueChanged, 50); // Wait for other plugins to handle copying stuff around
        }
    }
    _.each(options.selectors, function (sel) {
        var element = AJS.$(sel);
        element.on('input', callValueChanged);
        element.on('change', callValueChanged);
        element.on('selected', callValueChanged);
        element.on('unselect', callValueChanged);
    });
    valueChanged();
};

GH.StartWizardView.registerWizardStepPreRenderCallback = function (callback) {
    GH.StartWizardView.wizardStepPreRenderCallbacks.push(callback);
};

GH.StartWizardView.registerWizardStepPostRenderCallback = function (callback) {
    GH.StartWizardView.wizardStepPostRenderCallbacks.push(callback);
};

GH.StartWizardView.registerWizardStepOnValidateCallback = function (callback) {
    GH.StartWizardView.wizardStepOnValidateCallbacks.push(callback);
};

GH.StartWizardView.registerWizardStepOnCreateCallbacks = function (callback) {
    GH.StartWizardView.wizardStepOnCreateCallbacks.push(callback);
};