/* global GH.GridDataController */

define('jira-agile/rapid/ui/work/work-transition-service', ['jquery', 'underscore'], function ($, _) {
    "use strict";

    var TOO_MANY_PROJECTS_ERROR = 'gh.rapid.board.transition.too.many.projects';
    var MAX_PROJECT_NUMBER = 50;

    var WorkTransitionService = {};

    WorkTransitionService.dataPromise = $.Deferred();
    WorkTransitionService.data = null;
    WorkTransitionService.tooManyProjects = false;

    /**
     * Get data from server and set to local variable
     */
    WorkTransitionService.preloadData = function (projects) {
        if (!projects || projects.length > MAX_PROJECT_NUMBER) {
            return;
        }
        var projectIds = projects.map(function (project) {
            return project.id;
        });
        this.dataPromise = GH.Ajax.get({
            deferErrorHandling: true,
            url: '/xboard/work/transitions.json',
            data: { projectId: projectIds }
        }, 'workTransitionService').done(function (data) {
            WorkTransitionService.data = data;
        }).fail(function (data) {
            WorkTransitionService.tooManyProjects = data.error.errors[0].message === TOO_MANY_PROJECTS_ERROR;
        });
    };

    /**
     * Get transition data, if it hasn't gotten the data yet, will call the rest to load data
     * @return promise
     */
    WorkTransitionService.getData = function () {
        if (this.dataPromise.isRejected()) {
            this.preloadData();
        }
        return this.dataPromise;
    };

    /**
     * Query for possible transition of the given issueKey in the column
     * @param issueKey key of the issue needs querying
     * @param column the column data
     * @return promise
     */
    WorkTransitionService.queryTransitionsForColumn = function (issueKey, column) {
        var getTransitionForColumnPromise = $.Deferred();
        this.getData().done(function () {
            var possibleTransitionsForIssue = findPossibleTransitionForIssue(issueKey);
            getTransitionForColumnPromise.resolve(getTransitionsForColumn(possibleTransitionsForIssue, column));
        }).fail(getTransitionForColumnPromise.reject);
        return getTransitionForColumnPromise;
    };

    /**
     * Get transition data from the given transition id. This requires the data has been loaded.
     * Transition id is not unique so that we have to pass in issue key to get the correct workflow associated with the issue
     * and then get the transition from that.
     * @param issueKey key of the issue
     * @param transitionId id of the transition
     * @return transition object
     */
    WorkTransitionService.getTransitionById = function (issueKey, transitionId) {
        var workflowName = getWorkflowNameFromIssueKey(issueKey);
        return _.findWhere(this.data.workflowToTransitions[workflowName], { transitionId: Number(transitionId) });
    };

    WorkTransitionService.hasAnyTransitionHasScreen = function (issueKeys, transitionId) {
        return issueKeys.some(function (issueKey) {
            var transition = WorkTransitionService.getTransitionById(issueKey, transitionId);
            return transition.hasScreen;
        });
    };

    WorkTransitionService.hasAnyConditions = function (issueKey) {
        var transitions = findPossibleTransitionForIssue(issueKey);
        return transitions.some(function (transition) {
            return transition.hasConditions;
        });
    };

    var findPossibleTransitionForIssue = function findPossibleTransitionForIssue(issueKey) {
        var workflowName = getWorkflowNameFromIssueKey(issueKey);
        var transitions = WorkTransitionService.data.workflowToTransitions[workflowName];
        if (_.isUndefined(transitions)) {
            return [];
        }

        var issueData = getIssueDataByKey(issueKey);
        return transitions.filter(function (transition) {
            return !transition.isInitial && (_.isUndefined(transition.fromStatusId) || transition.fromStatusId === Number(issueData.statusId));
        });
    };

    var getTransitionsForColumn = function getTransitionsForColumn(transitions, columnData) {
        return transitions.filter(function (transition) {
            return columnData.statusIds.indexOf(String(transition.toStatusId)) > -1;
        });
    };

    var getIssueDataByKey = function getIssueDataByKey(issueKey) {
        var model = GH.GridDataController.getModel();
        return model.getIssueDataByKey(issueKey);
    };

    var getWorkflowNameFromIssueKey = function getWorkflowNameFromIssueKey(issueKey) {
        var issue = getIssueDataByKey(issueKey);
        var issueTypesToWorkflow = WorkTransitionService.data.projectAndIssueTypeToWorkflow[issue.projectId];
        return issueTypesToWorkflow[issue.typeId];
    };

    return WorkTransitionService;
});