/* global QUnit, require, sinon, ok, equal, deepEqual, test, asyncTest, expect, module */
AJS.test.require(['com.pyxis.greenhopper.jira:gh-test-common', 'com.pyxis.greenhopper.jira:gh-rapid-config', 'com.pyxis.greenhopper.jira:onboarding-create-project-component'], function() {

    var $;
    var _;
    var CreateProjectView;
    try {
        $ = require('jquery');
        _ = require('underscore');
        CreateProjectView = require('jira-agile/onboarding/create-project-sequence-view');
    } catch (e) {
        AJS.log('Was not able to load the resource, likely testing on an old version of JIRA');
    }

    //As these qunit tests cannot be put in dynamic module it must be included in all versions of JIRA, therefore,
    //It will try and run it in versions of JIRA which are not possible. If it does, do not fail these tests.
    if (!CreateProjectView) {
        module('CreateProjectSequenceView tests');
        test('Make sure there is at least one test for JIRA versions < 6.4', function () {
            ok(true);
        });

        return;
    }

    //
    //And now for the actual tests.
    //

    function stubProject(name, key) {
        return {
            name: name,
            key: key
        };
    }

    var SAMPLE_PROJECTS = [
        stubProject('Test Project', 'TEST'),
        stubProject('Another', 'ANO')
    ];

    var STUB_PROJECT_NAME = 'Stash';
    var STUB_PROJECT_KEY = 'STA';
    var STUB_LEAD_NAME = 'admin';
    var AGILE_PROJECT_TEMPLATE_KEY = 'com.pyxis.greenhopper.jira:gh-scrum-template';


    /**
     * Split a query list into the seperate key/value pairs
     * @param {String} request
     * @returns {Object}
     */
    function getPostParams(request) {
        var params = {};
        _.each(request.requestBody.split('&'), function(entry) {
            var entryValues = entry.split('=');
            params[entryValues[0]] = entryValues[1];
        });

        return params;
    }

    function change(selector, value) {
        var $el = this.$el.find(selector);
        $el.focus();
        $el.val(value);
        $el.change();
        // simulate a setTimeout because 'jira/field/create-project-field' has a
        // hardcoded 100ms delay until it triggers a change event
        this.clock.tick(200);
    }

    function createProject(name, key) {
        this.change('input[name=key]', key);
        this.change('input[name=name]', name);
    }


    module('CreateProjectSequenceView construction', {
        setup: function() {
            this.$el = $('<div/>');
            $('#qunit-fixture').append(this.$el);
        }
    });

    test('focuses the first input field upon creating the view', function() {
        var activeElement = document.activeElement;
        var view = new CreateProjectView({
            el: this.$el,
            leadName: STUB_LEAD_NAME
        });
        ok(document.activeElement === activeElement);
        view.render();
        ok(document.activeElement !== activeElement, 'active element should have changed upon rendering the view');
        ok(document.activeElement === view.$('input:first')[0], 'should be the first input element in the view');
    });


    module('CreateProjectSequenceView tests', {
        setup: function() {
            GH.Test.setUpFakeServer(this);
            this.$el = $('<div/>');
            $('#qunit-fixture').append(this.$el);

            this.server.respondWith([200, { 'Content-Type': 'application/json' }, JSON.stringify(SAMPLE_PROJECTS)]);
            this.createProjectSequenceView = new CreateProjectView({
                el: this.$el,
                leadName: STUB_LEAD_NAME
            });
            this.createProjectSequenceView.render();
            this.clock.tick(100);

            this.createProject = _.bind(createProject, this);
            this.change = _.bind(change, this);
        },
        teardown: function() {
            this.$el.remove();
            GH.Test.restoreServer(this);
        }
    });

    test('Test submit button is disabled at the beginning', function () {
        ok($('input[type=submit]', this.$el).attr('disabled'), 'Should be disabled');
        equal($('input[type=submit]', this.$el).attr('aria-disabled'), 'true', 'Should be aria disabled');
    });

    test('Test setting project name as current project has error', function () {
        this.server.respond();

        ok(!this.$el.find('.error').is(':visible'), 'Should be no error messages');

        this.change('input[name=name]', SAMPLE_PROJECTS[0].name);

        var errors = this.$el.find('.error');
        ok(errors.is(':visible'), 'Should have an error message');
    });

    test('Test a key has been assigned when changing name', function () {
        this.server.respond();

        this.change('input[name=name]', STUB_PROJECT_NAME);

        var keyValue = this.$el.find('[name=key]').val();
        ok(keyValue.length > 0, 'Key should have been assigned');
    });

    test('Already used key should show an error', function () {
        this.server.respond();
        this.change('input[name=key]', STUB_PROJECT_KEY);

        var validationResponse = {
            errors: {
                'projectKey': 'errorMessage'
            }
        };

        this.server.requests.pop().respond(200, { 'Content-Type': 'application/json' }, JSON.stringify(validationResponse));

        var errors = this.$el.find('.error');
        ok(errors.is(':visible'), 'Should have an error message');
    });

    test('Valid key does not show error', function () {
        this.server.respond();
        this.change('input[name=key]', STUB_PROJECT_KEY);

        this.server.requests.pop().respond(200, { 'Content-Type': 'application/json' }, JSON.stringify('{}'));

        var errors = this.$el.find('.error');
        ok(!errors.is(':visible'), 'Should have no error message');
    });

    test('Valid name and key should enable the submit button', function () {
        this.createProject(STUB_PROJECT_NAME, STUB_PROJECT_KEY);

        ok(!$('input[type=submit]', this.$el).attr('disabled'), 'Should not be disabled');
        equal($('input[type=submit]', this.$el).attr('aria-disabled'), 'false', 'Should not be aria disabled');
    });

    test('Project creation fields should be assigned correctly', function () {
        this.createProject(STUB_PROJECT_NAME, STUB_PROJECT_KEY);

        $('input[type=submit]', this.$el).click();

        // first request should check for a valid websudo session (if necessary)
        equal(this.server.requests.length, 3, 'Should have another request on a submit');

        var expectedParams = {
            name: STUB_PROJECT_NAME,
            key: STUB_PROJECT_KEY,
            lead: STUB_LEAD_NAME,
            projectTemplateWebItemKey: encodeURIComponent(AGILE_PROJECT_TEMPLATE_KEY)
        };
        // second request will create the project by POSTing data
        var params = getPostParams(this.server.requests.pop());

        deepEqual(params, expectedParams);

    });

    test('Failed project creation 400 shows error messages', function () {
        this.createProject(STUB_PROJECT_NAME, STUB_PROJECT_KEY);

        $('input[type=submit]', this.$el).click();

        var submitResponse = {
            errors: {
                projectKey: 'Some error'
            }
        };

        this.server.requests[this.server.requests.length - 1].respond(400, { 'Content-Type': 'application/json' }, JSON.stringify(submitResponse));

        ok(this.$el.find('.form-errors .title').is(':visible'), 'Should have error message');
    });

    test('Failed project creation other shows server error message', function () {
        this.createProject(STUB_PROJECT_NAME, STUB_PROJECT_KEY);

        $('input[type=submit]', this.$el).click();

        var submitResponse = {
            errors: {
                projectKey: 'Some error'
            }
        };
        this.server.requests[this.server.requests.length - 1].respond(500, { 'Content-Type': 'application/json' }, JSON.stringify(submitResponse));

        ok(this.$el.find('.form-errors .title').is(':visible'), 'Should have error message');
    });

    test('Unauthorised shows a link to login with the correct redirect url', function () {
        this.createProject(STUB_PROJECT_NAME, STUB_PROJECT_KEY);

        $('input[type=submit]', this.$el).click();

        this.server.requests[this.server.requests.length - 1].respond(401, { 'Content-Type': 'application/json' }, '{}');

        ok(this.$el.find('.form-errors .title').is(':visible'), 'Should have error message');
    });

    asyncTest('Project successful creation triggers event with redirect url', function () {
        expect(1);

        var redirectUrl = '/test';

        this.createProjectSequenceView.bind('done', function (response) {
            equal(response.redirectUrl, redirectUrl, 'Should have returned redirect url on finish');
            QUnit.start();
        });

        this.createProject(STUB_PROJECT_NAME, STUB_PROJECT_KEY);

        $('input[type=submit]', this.$el).click();

        var submitResponse = {
            redirectUrl: redirectUrl
        };
        this.server.requests[this.server.requests.length - 1].respond(200, { 'Content-Type': 'application/json' }, JSON.stringify(submitResponse));
    });

    asyncTest('Skipping project creation creates event', function () {
        expect(1);

        this.createProjectSequenceView.bind('skip', function () {
            ok(true);
            QUnit.start();
        });

        this.$el.find('#skip').click();
    });
});