/*
 * Copyright 2014 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package com.couchbase.client.deps.io.netty.example.spdy.server;

import com.couchbase.client.deps.io.netty.buffer.ByteBuf;
import com.couchbase.client.deps.io.netty.buffer.Unpooled;
import com.couchbase.client.deps.io.netty.channel.ChannelFutureListener;
import com.couchbase.client.deps.io.netty.channel.ChannelHandlerContext;
import com.couchbase.client.deps.io.netty.channel.SimpleChannelInboundHandler;
import com.couchbase.client.deps.io.netty.handler.codec.http.DefaultFullHttpResponse;
import com.couchbase.client.deps.io.netty.handler.codec.http.FullHttpResponse;
import com.couchbase.client.deps.io.netty.handler.codec.http.HttpRequest;
import com.couchbase.client.deps.io.netty.util.CharsetUtil;

import java.util.Date;

import static com.couchbase.client.deps.io.netty.handler.codec.http.HttpHeaderNames.CONNECTION;
import static com.couchbase.client.deps.io.netty.handler.codec.http.HttpHeaderNames.CONTENT_LENGTH;
import static com.couchbase.client.deps.io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;
import static com.couchbase.client.deps.io.netty.handler.codec.http.HttpHeaderValues.CLOSE;
import static com.couchbase.client.deps.io.netty.handler.codec.http.HttpHeaderValues.KEEP_ALIVE;
import static com.couchbase.client.deps.io.netty.handler.codec.http.HttpResponseStatus.CONTINUE;
import static com.couchbase.client.deps.io.netty.handler.codec.http.HttpResponseStatus.OK;
import static com.couchbase.client.deps.io.netty.handler.codec.http.HttpVersion.HTTP_1_0;
import static com.couchbase.client.deps.io.netty.handler.codec.http.HttpVersion.HTTP_1_1;
import static com.couchbase.client.deps.io.netty.handler.codec.http.HttpUtil.is100ContinueExpected;
import static com.couchbase.client.deps.io.netty.handler.codec.http.HttpUtil.isKeepAlive;

/**
 * HTTP handler that responds with a "Hello World"
 */
public class SpdyServerHandler extends SimpleChannelInboundHandler<Object> {

    @Override
    public void channelRead0(ChannelHandlerContext ctx, Object msg) throws Exception {
        if (msg instanceof HttpRequest) {
            HttpRequest req = (HttpRequest) msg;

            if (is100ContinueExpected(req)) {
                ctx.write(new DefaultFullHttpResponse(HTTP_1_1, CONTINUE, Unpooled.EMPTY_BUFFER));
            }
            boolean keepAlive = isKeepAlive(req);

            ByteBuf content = Unpooled.copiedBuffer("Hello World " + new Date(), CharsetUtil.UTF_8);

            FullHttpResponse response = new DefaultFullHttpResponse(HTTP_1_1, OK, content);
            response.headers().set(CONTENT_TYPE, "text/plain; charset=UTF-8");
            response.headers().setInt(CONTENT_LENGTH, response.content().readableBytes());

            if (keepAlive) {
                if (req.protocolVersion().equals(HTTP_1_0)) {
                    response.headers().set(CONNECTION, KEEP_ALIVE);
                }
                ctx.write(response);
            } else {
                // Tell the client we're going to close the connection.
                response.headers().set(CONNECTION, CLOSE);
                ctx.write(response).addListener(ChannelFutureListener.CLOSE);
            }
        }
    }

    @Override
    public void channelReadComplete(ChannelHandlerContext ctx) {
        ctx.flush();
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        cause.printStackTrace();
        ctx.close();
    }
}
