/*
 * Decompiled with CFR 0.152.
 */
package com.databricks.jdbc.api.impl.batch;

import com.databricks.jdbc.api.impl.batch.BatchCommand;
import com.databricks.jdbc.api.impl.batch.ImmutableBatchCommand;
import com.databricks.jdbc.exception.DatabricksBatchUpdateException;
import com.databricks.jdbc.exception.DatabricksValidationException;
import com.databricks.jdbc.log.JdbcLogger;
import com.databricks.jdbc.log.JdbcLoggerFactory;
import com.databricks.jdbc.model.telemetry.enums.DatabricksDriverErrorCode;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class DatabricksBatchExecutor {
    private static final JdbcLogger LOGGER = JdbcLoggerFactory.getLogger(DatabricksBatchExecutor.class);
    final Statement parentStatement;
    final List<BatchCommand> commands = new ArrayList<BatchCommand>();
    final int maxBatchSize;

    public DatabricksBatchExecutor(Statement parentStatement, int maxBatchSize) {
        this.parentStatement = parentStatement;
        this.maxBatchSize = maxBatchSize;
    }

    public void addCommand(String sql) throws DatabricksValidationException {
        if (sql == null) {
            throw new DatabricksValidationException("SQL command cannot be null");
        }
        if (this.commands.size() >= this.maxBatchSize) {
            throw new DatabricksValidationException("Batch size limit exceeded. Maximum allowed is " + this.maxBatchSize);
        }
        this.commands.add(ImmutableBatchCommand.builder().sql(sql).build());
    }

    public void clearCommands() {
        this.commands.clear();
    }

    public long[] executeBatch() throws DatabricksBatchUpdateException {
        if (this.commands.isEmpty()) {
            LOGGER.warn("No commands to execute in the batch");
            return new long[0];
        }
        long[] updateCounts = new long[this.commands.size()];
        Instant batchStartTime = Instant.now();
        try {
            for (int i = 0; i < this.commands.size(); ++i) {
                BatchCommand command = this.commands.get(i);
                Instant commandStartTime = Instant.now();
                try {
                    LOGGER.debug("Executing batch command {}: {}", i, command.getSql());
                    boolean hasResultSet = this.parentStatement.execute(command.getSql());
                    long updateCount = this.parentStatement.getLargeUpdateCount();
                    this.logCommandExecutionTime(i, commandStartTime, true);
                    if (hasResultSet) {
                        String message = String.format("Command %d in the batch attempted to return a ResultSet", i);
                        this.handleBatchFailure(updateCounts, i, batchStartTime, message, null);
                        continue;
                    }
                    if (updateCount != -1L) {
                        updateCounts[i] = updateCount;
                        continue;
                    }
                    updateCounts[i] = -2L;
                    continue;
                }
                catch (SQLException e) {
                    this.logCommandExecutionTime(i, commandStartTime, false);
                    LOGGER.error(e, "Error executing batch command at index {}: {}", i, e.getMessage());
                    String message = String.format("Batch execution failed at command %d: %s", i, e.getMessage());
                    this.handleBatchFailure(updateCounts, i, batchStartTime, message, e);
                }
            }
            this.clearCommands();
            Duration batchDuration = Duration.between(batchStartTime, Instant.now());
            LOGGER.debug("Total batch execution time: {} ms", batchDuration.toMillis());
            return updateCounts;
        }
        catch (DatabricksBatchUpdateException e) {
            LOGGER.error(e, "BatchUpdateException occurred: {}", e.getMessage());
            throw e;
        }
    }

    void logCommandExecutionTime(int commandIndex, Instant commandStartTime, boolean success) {
        Instant commandEndTime = Instant.now();
        Duration commandDuration = Duration.between(commandStartTime, commandEndTime);
        String status = success ? "executed" : "failed after";
        LOGGER.debug("Command {} {} {} ms", commandIndex, status, commandDuration.toMillis());
    }

    void handleBatchFailure(long[] updateCounts, int commandIndex, Instant batchStartTime, String message, SQLException cause) throws DatabricksBatchUpdateException {
        long[] countsSoFar = Arrays.copyOf(updateCounts, commandIndex);
        this.clearCommands();
        Duration batchDuration = Duration.between(batchStartTime, Instant.now());
        LOGGER.debug("Total batch execution time until failure: {} ms", batchDuration.toMillis());
        DatabricksBatchUpdateException exception = cause != null ? new DatabricksBatchUpdateException(message, cause.getSQLState(), cause.getErrorCode(), countsSoFar, (Throwable)cause) : new DatabricksBatchUpdateException(message, DatabricksDriverErrorCode.BATCH_EXECUTE_EXCEPTION, countsSoFar);
        throw exception;
    }
}

