/*
 * Decompiled with CFR 0.152.
 */
package org.tinyradius.packet;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tinyradius.attribute.Attributes;
import org.tinyradius.attribute.RadiusAttribute;
import org.tinyradius.dictionary.Dictionary;
import org.tinyradius.packet.RadiusPacket;
import org.tinyradius.util.RadiusPacketException;

public class AccessRequest
extends RadiusPacket {
    private static final Logger logger = LoggerFactory.getLogger(AccessRequest.class);
    private static final SecureRandom random = new SecureRandom();
    public static final String AUTH_PAP = "pap";
    public static final String AUTH_CHAP = "chap";
    public static final String AUTH_MS_CHAP_V2 = "mschapv2";
    public static final String AUTH_EAP = "eap";
    public static final Set<String> AUTH_PROTOCOLS = new HashSet<String>(Arrays.asList("pap", "chap", "mschapv2", "eap"));
    private String authProtocol = "pap";
    private transient String password;
    private transient byte[] chapPassword;
    private transient byte[] chapChallenge;
    private static final int USER_NAME = 1;
    private static final int USER_PASSWORD = 2;
    private static final int CHAP_PASSWORD = 3;
    private static final int CHAP_CHALLENGE = 60;
    private static final int EAP_MESSAGE = 79;
    private static final int MICROSOFT = 311;
    private static final int MS_CHAP_CHALLENGE = 11;
    private static final int MS_CHAP2_RESPONSE = 25;
    private static final int MS_CHAP_RESPONSE = 1;

    public AccessRequest(Dictionary dictionary, int identifier, byte[] authenticator) {
        this(dictionary, identifier, authenticator, new ArrayList<RadiusAttribute>());
    }

    public AccessRequest(Dictionary dictionary, int identifier, byte[] authenticator, List<RadiusAttribute> attributes) {
        super(dictionary, 1, identifier, authenticator, attributes);
    }

    public AccessRequest(Dictionary dictionary, int identifier, byte[] authenticator, String userName, String userPassword) {
        this(dictionary, identifier, authenticator);
        this.setUserName(userName);
        this.setUserPassword(userPassword);
    }

    public void setUserName(String userName) {
        Objects.requireNonNull(userName, "User name not set");
        if (userName.isEmpty()) {
            throw new IllegalArgumentException("Empty user name not allowed");
        }
        this.removeAttributes(1);
        this.addAttribute(Attributes.createAttribute(this.getDictionary(), -1, 1, userName));
    }

    public void setUserPassword(String userPassword) {
        Objects.requireNonNull(userPassword, "User password not set");
        if (userPassword.isEmpty()) {
            throw new IllegalArgumentException("Password is empty");
        }
        this.password = userPassword;
    }

    public String getUserPassword() {
        return this.password;
    }

    public String getUserName() {
        RadiusAttribute attribute = this.getAttribute(1);
        return attribute == null ? null : attribute.getValueString();
    }

    public String getAuthProtocol() {
        return this.authProtocol;
    }

    public void setAuthProtocol(String authProtocol) {
        if (authProtocol == null || !AUTH_PROTOCOLS.contains(authProtocol)) {
            throw new IllegalArgumentException("protocol must be in " + AUTH_PROTOCOLS);
        }
        this.authProtocol = authProtocol;
    }

    @Override
    public void verify(String sharedSecret, byte[] ignored) throws RadiusPacketException {
        if (!this.decryptPasswords(sharedSecret)) {
            throw new RadiusPacketException("Access-Request: User-Password or CHAP-Password/CHAP-Challenge missing");
        }
    }

    public boolean decryptPasswords(String sharedSecret) throws RadiusPacketException {
        RadiusAttribute userPassword = this.getAttribute(2);
        if (userPassword != null) {
            this.setAuthProtocol(AUTH_PAP);
            this.password = this.decodePapPassword(userPassword.getValue(), sharedSecret.getBytes(StandardCharsets.UTF_8));
            return true;
        }
        RadiusAttribute chapPassword = this.getAttribute(3);
        RadiusAttribute chapChallenge = this.getAttribute(60);
        if (chapPassword != null) {
            this.setAuthProtocol(AUTH_CHAP);
            this.chapPassword = chapPassword.getValue();
            this.chapChallenge = chapChallenge != null ? chapChallenge.getValue() : this.getAuthenticator();
            return true;
        }
        RadiusAttribute msChapChallenge = this.getAttribute(311, 11);
        RadiusAttribute msChap2Response = this.getAttribute(311, 25);
        if (msChap2Response == null) {
            msChap2Response = this.getAttribute(311, 1);
        }
        if (msChapChallenge != null && msChap2Response != null) {
            this.setAuthProtocol(AUTH_MS_CHAP_V2);
            this.chapPassword = msChap2Response.getValue();
            this.chapChallenge = msChapChallenge.getValue();
            return true;
        }
        List<RadiusAttribute> eapMessage = this.getAttributes(79);
        if (eapMessage.size() > 0) {
            this.setAuthProtocol(AUTH_EAP);
            return true;
        }
        return false;
    }

    public boolean verifyPassword(String plaintext) throws UnsupportedOperationException {
        if (plaintext == null || plaintext.isEmpty()) {
            throw new IllegalArgumentException("password is empty");
        }
        switch (this.getAuthProtocol()) {
            case "chap": {
                return this.verifyChapPassword(plaintext);
            }
            case "mschapv2": {
                throw new UnsupportedOperationException("mschapv2 verification not supported yet");
            }
            case "eap": {
                throw new UnsupportedOperationException("eap verification not supported yet");
            }
        }
        return this.getUserPassword().equals(plaintext);
    }

    @Override
    public AccessRequest encodeRequest(String sharedSecret) throws UnsupportedOperationException {
        if (sharedSecret == null || sharedSecret.isEmpty()) {
            throw new IllegalArgumentException("shared secret cannot be null/empty");
        }
        byte[] newAuthenticator = this.getAuthenticator() == null ? this.random16bytes() : this.getAuthenticator();
        AccessRequest accessRequest = new AccessRequest(this.getDictionary(), this.getIdentifier(), newAuthenticator, new ArrayList<RadiusAttribute>(this.getAttributes()));
        this.copyTransientFields(accessRequest);
        this.encodeRequestAttributes(newAuthenticator, sharedSecret).forEach(a -> {
            accessRequest.removeAttributes(a.getType());
            accessRequest.addAttribute((RadiusAttribute)a);
        });
        return accessRequest;
    }

    @Override
    public RadiusPacket encodeResponse(String sharedSecret, byte[] requestAuthenticator) {
        throw new UnsupportedOperationException();
    }

    protected List<RadiusAttribute> encodeRequestAttributes(byte[] authenticator, String sharedSecret) throws UnsupportedOperationException {
        if (this.password != null && !this.password.isEmpty()) {
            switch (this.getAuthProtocol()) {
                case "pap": {
                    return Collections.singletonList(Attributes.createAttribute(this.getDictionary(), -1, 2, this.encodePapPassword(authenticator, this.password.getBytes(StandardCharsets.UTF_8), sharedSecret.getBytes(StandardCharsets.UTF_8))));
                }
                case "chap": {
                    byte[] challenge = this.random16bytes();
                    return Arrays.asList(Attributes.createAttribute(this.getDictionary(), -1, 60, challenge), Attributes.createAttribute(this.getDictionary(), -1, 3, this.computeChapPassword((byte)random.nextInt(256), this.password, challenge)));
                }
                case "mschapv2": {
                    throw new UnsupportedOperationException("Encoding not supported for mschapv2");
                }
                case "eap": {
                    throw new UnsupportedOperationException("Encoding not supported for eap");
                }
            }
        }
        return Collections.emptyList();
    }

    private byte[] encodePapPassword(byte[] authenticator, byte[] userPass, byte[] sharedSecret) {
        Objects.requireNonNull(userPass, "userPass cannot be null");
        Objects.requireNonNull(sharedSecret, "sharedSecret cannot be null");
        byte[] ciphertext = authenticator;
        byte[] P = AccessRequest.pad(userPass);
        ByteBuffer buffer = ByteBuffer.allocate(P.length);
        for (int i = 0; i < P.length; i += 16) {
            ciphertext = AccessRequest.xor16(P, i, this.md5(sharedSecret, ciphertext));
            buffer.put(ciphertext);
        }
        return buffer.array();
    }

    private String decodePapPassword(byte[] encryptedPass, byte[] sharedSecret) throws RadiusPacketException {
        if (encryptedPass.length < 16) {
            logger.warn("Malformed packet: User-Password attribute length must be greater than 15, actual {}", (Object)encryptedPass.length);
            throw new RadiusPacketException("Malformed User-Password attribute");
        }
        ByteBuffer buffer = ByteBuffer.allocate(encryptedPass.length);
        byte[] ciphertext = this.getAuthenticator();
        for (int i = 0; i < encryptedPass.length; i += 16) {
            buffer.put(AccessRequest.xor16(encryptedPass, i, this.md5(sharedSecret, ciphertext)));
            ciphertext = Arrays.copyOfRange(encryptedPass, i, 16);
        }
        return AccessRequest.stripNullPadding(new String(buffer.array(), StandardCharsets.UTF_8));
    }

    private byte[] computeChapPassword(byte chapId, String plaintextPw, byte[] chapChallenge) {
        MessageDigest md5 = AccessRequest.getMd5Digest();
        md5.update(chapId);
        md5.update(plaintextPw.getBytes(StandardCharsets.UTF_8));
        md5.update(chapChallenge);
        return ByteBuffer.allocate(17).put(chapId).put(md5.digest()).array();
    }

    private boolean verifyChapPassword(String plaintext) {
        if (plaintext == null || plaintext.isEmpty()) {
            logger.warn("plaintext must not be empty");
        } else if (this.chapChallenge == null) {
            logger.warn("CHAP challenge is null");
        } else if (this.chapPassword == null || this.chapPassword.length != 17) {
            logger.warn("CHAP password must be 17 bytes");
        } else {
            return Arrays.equals(this.chapPassword, this.computeChapPassword(this.chapPassword[0], plaintext, this.chapChallenge));
        }
        return false;
    }

    private byte[] md5(byte[] a, byte[] b) {
        MessageDigest md = AccessRequest.getMd5Digest();
        md.update(a);
        return md.digest(b);
    }

    private byte[] random16bytes() {
        byte[] randomBytes = new byte[16];
        random.nextBytes(randomBytes);
        return randomBytes;
    }

    private static byte[] xor16(byte[] src1, int src1offset, byte[] src2) {
        byte[] dst = new byte[16];
        Objects.requireNonNull(src1, "src1 is null");
        Objects.requireNonNull(src2, "src2 is null");
        if (src1offset < 0) {
            throw new IndexOutOfBoundsException("src1offset is less than 0");
        }
        if (src1offset + 16 > src1.length) {
            throw new IndexOutOfBoundsException("bytes in src1 is less than src1offset plus 16");
        }
        if (16 > src2.length) {
            throw new IndexOutOfBoundsException("bytes in src2 is less than 16");
        }
        for (int i = 0; i < 16; ++i) {
            dst[i] = (byte)(src1[i + src1offset] ^ src2[i]);
        }
        return dst;
    }

    static byte[] pad(byte[] val) {
        Objects.requireNonNull(val, "value cannot be null");
        int length = Math.max((int)(Math.ceil((double)val.length / 16.0) * 16.0), 16);
        byte[] padded = new byte[length];
        System.arraycopy(val, 0, padded, 0, val.length);
        return padded;
    }

    private static String stripNullPadding(String s) {
        int i = s.indexOf(0);
        return i > 0 ? s.substring(0, i) : s;
    }

    private AccessRequest copyTransientFields(AccessRequest target) {
        target.password = this.password;
        target.chapPassword = this.chapPassword;
        target.chapChallenge = this.chapChallenge;
        return target;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof AccessRequest)) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        AccessRequest that = (AccessRequest)o;
        return Objects.equals(this.authProtocol, that.authProtocol) && Objects.equals(this.password, that.password) && Arrays.equals(this.chapPassword, that.chapPassword) && Arrays.equals(this.chapChallenge, that.chapChallenge);
    }

    @Override
    public int hashCode() {
        int result = Objects.hash(super.hashCode(), this.authProtocol, this.password);
        result = 31 * result + Arrays.hashCode(this.chapPassword);
        result = 31 * result + Arrays.hashCode(this.chapChallenge);
        return result;
    }

    @Override
    public AccessRequest copy() {
        return this.copyTransientFields((AccessRequest)super.copy());
    }
}

