/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.syntax

import ksp.org.jetbrains.kotlin.KtLightSourceElement
import ksp.org.jetbrains.kotlin.KtNodeTypes
import ksp.org.jetbrains.kotlin.KtPsiSourceElement
import ksp.org.jetbrains.kotlin.KtRealSourceElementKind
import ksp.org.jetbrains.kotlin.KtSourceElement
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.analysis.forEachChildOfType
import ksp.org.jetbrains.kotlin.fir.declarations.FirConstructor
import ksp.org.jetbrains.kotlin.lexer.KtTokens
import ksp.org.jetbrains.kotlin.psi.KtConstructor
import ksp.org.jetbrains.kotlin.psi.KtDeclaration

object FirMissingConstructorKeywordSyntaxChecker : FirDeclarationSyntaxChecker<FirConstructor, KtDeclaration>() {

    override fun isApplicable(element: FirConstructor, source: KtSourceElement): Boolean = true

    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun checkPsi(
        element: FirConstructor,
        source: KtPsiSourceElement,
        psi: KtDeclaration,
    ) {
        if (psi is KtConstructor<*> && psi.modifierList != null && psi.getConstructorKeyword() == null) {
            reporter.reportOn(source, FirErrors.MISSING_CONSTRUCTOR_KEYWORD)
        }
    }

    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun checkLightTree(
        element: FirConstructor,
        source: KtLightSourceElement,
    ) {
        if (source.kind !is KtRealSourceElementKind) return
        var hasModifiers = false
        var hasConstructorKeyword = false
        source.forEachChildOfType(importantNodeTypes, depth = 1) {
            if (it.elementType == KtTokens.CONSTRUCTOR_KEYWORD) {
                hasConstructorKeyword = true
            } else {
                hasModifiers = true
            }
        }
        if (hasModifiers && !hasConstructorKeyword) {
            reporter.reportOn(source, FirErrors.MISSING_CONSTRUCTOR_KEYWORD)
        }
    }

    private val importantNodeTypes = setOf(KtNodeTypes.MODIFIER_LIST, KtTokens.CONSTRUCTOR_KEYWORD)
}
