/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.js.checkers.declaration

import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.FirBasicDeclarationChecker
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.js.FirJsErrors
import ksp.org.jetbrains.kotlin.fir.analysis.js.checkers.isExportedObject
import ksp.org.jetbrains.kotlin.fir.declarations.FirDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.FirMemberDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.FirTypeAlias
import ksp.org.jetbrains.kotlin.fir.declarations.expectForActual
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isActual
import ksp.org.jetbrains.kotlin.fir.isEnabled
import ksp.org.jetbrains.kotlin.fir.resolve.toClassSymbol
import ksp.org.jetbrains.kotlin.fir.types.abbreviatedTypeOrSelf
import ksp.org.jetbrains.kotlin.fir.types.coneType
import ksp.org.jetbrains.kotlin.resolve.multiplatform.ExpectActualMatchingCompatibility

object FirJsExportedActualMatchExpectChecker : FirBasicDeclarationChecker(MppCheckerKind.Platform) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirDeclaration) {
        if (
            !LanguageFeature.AllowExpectDeclarationsInJsExport.isEnabled() ||
            declaration !is FirMemberDeclaration ||
            !declaration.isActual
        ) return

        val expectForActualMatchingData = declaration.symbol.expectForActual.orEmpty()
        val correspondingExpectDeclaration =
            expectForActualMatchingData[ExpectActualMatchingCompatibility.MatchedSuccessfully]?.singleOrNull() ?: return

        val correspondingActualDeclaration = when (declaration) {
            is FirTypeAlias -> {
                declaration.expandedTypeRef.coneType.abbreviatedTypeOrSelf.toClassSymbol() ?: return
            }
            else -> declaration.symbol
        }


        if (correspondingExpectDeclaration.isExportedObject() && !correspondingActualDeclaration.isExportedObject()) {
            reporter.reportOn(declaration.source, FirJsErrors.NOT_EXPORTED_ACTUAL_DECLARATION_WHILE_EXPECT_IS_EXPORTED)
        }
    }
}