/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.light.classes.symbol.classes

import ksp.com.intellij.psi.*
import ksp.org.jetbrains.kotlin.analysis.api.KaImplementationDetail
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaModule
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaScriptSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.asJava.classes.*
import ksp.org.jetbrains.kotlin.asJava.elements.FakeFileForLightClass
import ksp.org.jetbrains.kotlin.light.classes.symbol.analyzeForLightClasses
import ksp.org.jetbrains.kotlin.light.classes.symbol.cachedValue
import ksp.org.jetbrains.kotlin.light.classes.symbol.fields.SymbolLightField
import ksp.org.jetbrains.kotlin.light.classes.symbol.methods.SymbolLightMethodForScriptDefaultConstructor
import ksp.org.jetbrains.kotlin.light.classes.symbol.methods.SymbolLightMethodForScriptMain
import ksp.org.jetbrains.kotlin.light.classes.symbol.modifierLists.InitializedModifiersBox
import ksp.org.jetbrains.kotlin.light.classes.symbol.modifierLists.SymbolLightClassModifierList
import ksp.org.jetbrains.kotlin.light.classes.symbol.withSymbol
import ksp.org.jetbrains.kotlin.psi.KtScript

internal class SymbolLightClassForScript private constructor(
    override val script: KtScript,
    private val symbolPointer: KaSymbolPointer<KaScriptSymbol>,
    ktModule: KaModule,
) : KtLightClassForScript, SymbolLightClassBase(ktModule, script.manager) {
    @OptIn(KaImplementationDetail::class)
    internal constructor(
        script: KtScript,
        ktModule: KaModule,
    ) : this(
        script,
        analyzeForLightClasses(ktModule) {
            script.symbol.createPointer()
        },
        ktModule,
    )

    private fun MutableList<PsiMethod>.addScriptDefaultMethods() {
        val defaultConstructor = SymbolLightMethodForScriptDefaultConstructor(
            script,
            this@SymbolLightClassForScript,
            METHOD_INDEX_FOR_DEFAULT_CTOR
        )
        add(defaultConstructor)

        val mainMethod = SymbolLightMethodForScriptMain(
            script,
            this@SymbolLightClassForScript,
            METHOD_INDEX_FOR_SCRIPT_MAIN
        )
        add(mainMethod)
    }

    override fun getOwnMethods(): List<PsiMethod> = cachedValue {
        val result = mutableListOf<PsiMethod>()

        result.addScriptDefaultMethods()

        symbolPointer.withSymbol(ktModule) { scriptSymbol ->
            createMethods(this@SymbolLightClassForScript, scriptSymbol.declaredMemberScope.callables, result)
        }
        result
    }

    override fun getOwnFields(): List<PsiField> = cachedValue {
        symbolPointer.withSymbol(ktModule) { scriptSymbol ->
            buildList {
                addPropertyBackingFields(this@SymbolLightClassForScript, this, scriptSymbol, SymbolLightField.FieldNameGenerator())
            }
        }
    }

    override fun getOwnInnerClasses(): List<SymbolLightClassBase> = cachedValue {
        symbolPointer.withSymbol(ktModule) { scriptSymbol ->
            createInnerClasses(scriptSymbol, manager, this@SymbolLightClassForScript, classOrObject = null)
        }
    }

    override fun copy(): SymbolLightClassForScript =
        SymbolLightClassForScript(script, symbolPointer, ktModule)

    override fun getModifierList(): PsiModifierList = cachedValue {
        SymbolLightClassModifierList(
            containingDeclaration = this@SymbolLightClassForScript,
            modifiersBox = InitializedModifiersBox(PsiModifier.PUBLIC, PsiModifier.FINAL)
        )
    }

    override fun hasModifierProperty(name: String): Boolean = modifierList.hasModifierProperty(name)

    private val _containingFile by lazyPub {
        FakeFileForLightClass(
            script.containingKtFile,
            lightClass = this,
            packageFqName = script.fqName.parent(),
        )
    }

    override fun getContainingFile() = _containingFile

    override fun getName() = script.fqName.shortName().asString()

    override fun equals(other: Any?): Boolean = this === other || other is SymbolLightClassForScript && other.script == script

    override fun hashCode(): Int = script.hashCode()

    override fun getNameIdentifier(): PsiIdentifier? = null

    override fun getContainingClass(): PsiClass? = null

    override fun isDeprecated(): Boolean = false

    override fun getTypeParameters(): Array<PsiTypeParameter> = PsiTypeParameter.EMPTY_ARRAY

    override fun getTypeParameterList(): PsiTypeParameterList? = null

    override fun getQualifiedName() = script.fqName.asString()

    override fun isInterface(): Boolean = false

    override fun isAnnotationType(): Boolean = false

    override fun isEnum(): Boolean = false

    private val _extendsList: PsiReferenceList by lazyPub {
        KotlinSuperTypeListBuilder(
            this,
            kotlinOrigin = kotlinOrigin?.getSuperTypeList(),
            manager = manager,
            language = language,
            role = PsiReferenceList.Role.EXTENDS_LIST,
        ).apply {
            addReference("kotlin.script.templates.standard.ScriptTemplateWithArgs")
        }
    }

    override fun getExtendsList(): PsiReferenceList = _extendsList

    private val _implementsList by lazyPub {
        KotlinSuperTypeListBuilder(
            this,
            kotlinOrigin = kotlinOrigin?.getSuperTypeList(),
            manager = manager,
            language = language,
            role = PsiReferenceList.Role.IMPLEMENTS_LIST,
        )
    }

    override fun getImplementsList(): PsiReferenceList = _implementsList

    override fun getInterfaces(): Array<PsiClass> = PsiClass.EMPTY_ARRAY

    override fun getSuperClass(): PsiClass? {
        return JavaPsiFacade.getInstance(project).findClass(CommonClassNames.JAVA_LANG_OBJECT, resolveScope)
    }

    override fun getSupers(): Array<PsiClass> {
        return superClass?.let { arrayOf(it) } ?: PsiClass.EMPTY_ARRAY
    }

    override fun getSuperTypes(): Array<PsiClassType> {
        return arrayOf(PsiType.getJavaLangObject(manager, resolveScope))
    }

    override fun getScope(): PsiElement = parent

    override fun isInheritorDeep(baseClass: PsiClass, classToByPass: PsiClass?): Boolean = false

}