/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.psi.stubs.impl

import ksp.com.intellij.psi.PsiElement
import ksp.com.intellij.psi.stubs.StubElement
import ksp.com.intellij.util.io.StringRef
import ksp.org.jetbrains.kotlin.psi.KtImplementationDetail
import ksp.org.jetbrains.kotlin.psi.stubs.KotlinStubElement
import ksp.org.jetbrains.kotlin.utils.exceptions.checkWithAttachment

object Utils {
    fun wrapStrings(names: List<String>): Array<StringRef> {
        return Array(names.size) { i -> StringRef.fromString(names[i])!! }
    }
}


/** Creates a deep copy of the given [this] */
@KtImplementationDetail
fun KotlinFileStubImpl.deepCopy(): KotlinFileStubImpl = copyStubRecursively(
    originalStub = this,
    newParentStub = null,
) as KotlinFileStubImpl

/**
 * Returns a copy of [originalStub].
 */
@OptIn(KtImplementationDetail::class)
private fun <T : PsiElement> copyStubRecursively(
    originalStub: StubElement<T>,
    newParentStub: StubElement<*>?,
): StubElement<*> {
    require(originalStub is KotlinStubElement<*>) {
        "${KotlinStubElement::class.simpleName} is expected, but ${originalStub::class.simpleName} is found"
    }

    val stubCopy = originalStub.copyInto(newParentStub)
    checkWithAttachment(
        originalStub::class == stubCopy::class,
        { "${originalStub::class.simpleName} is expected, but ${stubCopy::class.simpleName} is found" },
    )

    for (originalChild in originalStub.childrenStubs) {
        copyStubRecursively(originalStub = originalChild, newParentStub = stubCopy)
    }

    return stubCopy
}
