/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.decompiler.konan

import ksp.com.intellij.openapi.application.ApplicationManager
import ksp.com.intellij.openapi.vfs.VirtualFile
import ksp.com.intellij.util.containers.ContainerUtil
import ksp.org.jetbrains.kotlin.library.KLIB_MANIFEST_FILE_NAME
import ksp.org.jetbrains.kotlin.library.KLIB_METADATA_FILE_EXTENSION
import ksp.org.jetbrains.kotlin.library.KLIB_MODULE_METADATA_FILE_NAME
import ksp.org.jetbrains.kotlin.library.metadata.KlibMetadataProtoBuf
import ksp.org.jetbrains.kotlin.library.metadata.parseModuleHeader
import ksp.org.jetbrains.kotlin.library.metadata.parsePackageFragment
import ksp.org.jetbrains.kotlin.library.readKonanLibraryVersioning
import ksp.org.jetbrains.kotlin.metadata.ProtoBuf
import ksp.org.jetbrains.kotlin.metadata.deserialization.MetadataVersion
import java.io.IOException
import java.util.*

class KlibLoadingMetadataCache {
    // Use special CacheKey class instead of VirtualFile for cache keys. Certain types of VirtualFiles (for example, obtained from JarFileSystem)
    // do not compare path (url) and modification stamp in equals() method.
    private data class CacheKey(
        val url: String,
        val modificationStamp: Long
    ) {
        constructor(virtualFile: VirtualFile) : this(virtualFile.url, virtualFile.modificationStamp)
    }

    // ConcurrentWeakValueHashMap does not allow null values.
    private class CacheValue<T : Any>(val value: T?)

    private val packageFragmentCache = ContainerUtil.createConcurrentWeakValueMap<CacheKey, CacheValue<ProtoBuf.PackageFragment>>()
    private val moduleHeaderCache = ContainerUtil.createConcurrentWeakValueMap<CacheKey, CacheValue<KlibMetadataProtoBuf.Header>>()
    private val libraryMetadataVersionCache = ContainerUtil.createConcurrentWeakValueMap<CacheKey, CacheValue<MetadataVersion>>()

    fun getCachedPackageFragment(packageFragmentFile: VirtualFile): ProtoBuf.PackageFragment? {
        check(packageFragmentFile.extension == KLIB_METADATA_FILE_EXTENSION) {
            "Not a package metadata file: $packageFragmentFile"
        }

        return packageFragmentCache.computeIfAbsent(
            CacheKey(packageFragmentFile)
        ) {
            CacheValue(computePackageFragment(packageFragmentFile))
        }.value
    }

    fun getCachedModuleHeader(moduleHeaderFile: VirtualFile): KlibMetadataProtoBuf.Header? {
        check(moduleHeaderFile.name == KLIB_MODULE_METADATA_FILE_NAME) {
            "Not a module header file: $moduleHeaderFile"
        }

        return moduleHeaderCache.computeIfAbsent(
            CacheKey(moduleHeaderFile)
        ) {
            CacheValue(computeModuleHeader(moduleHeaderFile))
        }.value
    }

    fun getCachedPackageFragmentWithVersion(packageFragmentFile: VirtualFile): Pair<ProtoBuf.PackageFragment?, MetadataVersion?> {
        val packageFragment = getCachedPackageFragment(packageFragmentFile) ?: return null to null
        val version = getCachedMetadataVersion(getKlibLibraryRootForPackageFragment(packageFragmentFile))
        return packageFragment to version
    }

    private fun getCachedMetadataVersion(libraryRoot: VirtualFile): MetadataVersion? {
        val manifestFile = libraryRoot.findChild(KLIB_MANIFEST_FILE_NAME) ?: return null

        val metadataVersion = libraryMetadataVersionCache.computeIfAbsent(
            CacheKey(manifestFile)
        ) {
            CacheValue(computeLibraryMetadataVersion(manifestFile))
        }.value

        return metadataVersion
    }

    private fun getKlibLibraryRootForPackageFragment(packageFragmentFile: VirtualFile): VirtualFile {
        return packageFragmentFile.parent.parent.parent
    }

    private fun isMetadataCompatible(libraryRoot: VirtualFile): Boolean {
        val metadataVersion = getCachedMetadataVersion(libraryRoot) ?: return false

        return metadataVersion.isCompatibleWithCurrentCompilerVersion()
    }

    private fun computePackageFragment(packageFragmentFile: VirtualFile): ProtoBuf.PackageFragment? {
        if (!isMetadataCompatible(getKlibLibraryRootForPackageFragment(packageFragmentFile)))
            return null

        return try {
            parsePackageFragment(packageFragmentFile.contentsToByteArray(false))
        } catch (_: IOException) {
            null
        }
    }

    private fun computeModuleHeader(moduleHeaderFile: VirtualFile): KlibMetadataProtoBuf.Header? {
        if (!isMetadataCompatible(moduleHeaderFile.parent.parent))
            return null

        return try {
            parseModuleHeader(moduleHeaderFile.contentsToByteArray(false))
        } catch (_: IOException) {
            null
        }
    }

    private fun computeLibraryMetadataVersion(manifestFile: VirtualFile): MetadataVersion? = try {
        val versioning = Properties().apply { manifestFile.inputStream.use { load(it) } }.readKonanLibraryVersioning()
        versioning.metadataVersion
    } catch (_: IOException) {
        // ignore and cache null value
        null
    } catch (_: IllegalArgumentException) {
        // ignore and cache null value
        null
    }

    companion object {
        @JvmStatic
        fun getInstance(): KlibLoadingMetadataCache =
            ApplicationManager.getApplication().getService(KlibLoadingMetadataCache::class.java)
    }
}
