/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.buildtools.api.jvm

import ksp.org.jetbrains.kotlin.buildtools.api.ExperimentalBuildToolsApi
import ksp.org.jetbrains.kotlin.buildtools.api.SourcesChanges
import ksp.org.jetbrains.kotlin.buildtools.api.internal.BaseOption
import ksp.org.jetbrains.kotlin.buildtools.api.jvm.operations.JvmCompilationOperation
import java.nio.file.Path

/**
 * Represents a configuration that enables incremental compilation.
 *
 * This interface is not intended to be implemented by the API consumers.
 *
 * Currently, the only supported implementation of this interface is [JvmSnapshotBasedIncrementalCompilationConfiguration],
 * but more may be added in the future.
 *
 * @see JvmSnapshotBasedIncrementalCompilationConfiguration
 */
@ExperimentalBuildToolsApi
public interface JvmIncrementalCompilationConfiguration

/**
 * A configuration for incremental compilation based on snapshots.
 *
 * @property workingDirectory the working directory for the IC operation to store internal objects.
 * @property sourcesChanges changes in the source files, which can be unknown, to-be-calculated, or known.
 * @property dependenciesSnapshotFiles a list of paths to dependency snapshot files produced by [org.jetbrains.kotlin.buildtools.api.jvm.operations.JvmClasspathSnapshottingOperation].
 * @property options an option set produced by [JvmCompilationOperation.createSnapshotBasedIcOptions]
 */
@ExperimentalBuildToolsApi
public class JvmSnapshotBasedIncrementalCompilationConfiguration(
    public val workingDirectory: Path,
    public val sourcesChanges: SourcesChanges,
    public val dependenciesSnapshotFiles: List<Path>,
    public val shrunkClasspathSnapshot: Path,
    public val options: JvmSnapshotBasedIncrementalCompilationOptions,
) : JvmIncrementalCompilationConfiguration

/**
 * Options for [JvmSnapshotBasedIncrementalCompilationConfiguration].
 *
 * @since 2.3.0
 */
@ExperimentalBuildToolsApi
public interface JvmSnapshotBasedIncrementalCompilationOptions {
    /**
     * Base class for [JvmSnapshotBasedIncrementalCompilationOptions] options.
     *
     * @see get
     * @see set
     */
    public class Option<V> internal constructor(id: String) : BaseOption<V>(id)

    /**
     * Get the value for option specified by [key] if it was previously [set] or if it has a default value.
     *
     * @return the previously set value for an option
     * @throws IllegalStateException if the option was not set and has no default value
     */
    public operator fun <V> get(key: Option<V>): V

    /**
     * Set the [value] for option specified by [key], overriding any previous value for that option.
     */
    public operator fun <V> set(key: Option<V>, value: V)

    public companion object {

        /**
         * The root project directory, used for computing relative paths for source files in the incremental compilation caches.
         *
         * If it is not specified, incremental compilation caches will be non-relocatable.
         */
        @JvmField
        public val ROOT_PROJECT_DIR: Option<Path?> = Option("ROOT_PROJECT_DIR")

        /**
         * The build directory, used for computing relative paths for output files in the incremental compilation caches.
         *
         * If it is not specified, incremental compilation caches will be non-relocatable.
         */
        @JvmField
        public val MODULE_BUILD_DIR: Option<Path?> = Option("MODULE_BUILD_DIR")

        /**
         * Controls whether incremental compilation will analyze Java files precisely for better changes detection.
         */
        @JvmField
        public val PRECISE_JAVA_TRACKING: Option<Boolean> =
            Option("PRECISE_JAVA_TRACKING")

        /**
         * Controls whether incremental compilation should perform file-by-file backup of previously compiled files
         * and revert them in the case of a compilation failure
         */
        @JvmField
        public val BACKUP_CLASSES: Option<Boolean> = Option("BACKUP_CLASSES")

        /**
         * Controls whether caches should remain in memory
         * and not be flushed to the disk until the compilation can be marked as successful.
         */
        @JvmField
        public val KEEP_IC_CACHES_IN_MEMORY: Option<Boolean> = Option("KEEP_IC_CACHES_IN_MEMORY")

        /**
         * Controls whether the non-incremental mode of the incremental compiler is forced.
         * The non-incremental mode of the incremental compiler means that during the compilation,
         * the compiler will collect enough information to perform subsequent builds incrementally.
         */
        @JvmField
        public val FORCE_RECOMPILATION: Option<Boolean> = Option("FORCE_RECOMPILATION")

        /**
         * The directories that the compiler will clean in the case of fallback to non-incremental compilation.
         *
         * The default ones are calculated in the case of a `null` value as a set of the incremental compilation working directory
         * passed to [JvmSnapshotBasedIncrementalCompilationConfiguration] and the classes output directory from the compiler arguments.
         *
         * If the value is set explicitly, it must contain the above-mentioned default directories.
         */
        @JvmField
        public val OUTPUT_DIRS: Option<Set<Path>?> = Option("OUTPUT_DIRS")

        /**
         * Controls whether classpath snapshots comparing should be avoided.
         *
         * Can be used as an optimization if the check is already performed by the API consumer.
         */
        @JvmField
        public val ASSURED_NO_CLASSPATH_SNAPSHOT_CHANGES: Option<Boolean> =
            Option("ASSURED_NO_CLASSPATH_SNAPSHOT_CHANGES")

        /**
         * Controls whether the *experimental* incremental runner based on Kotlin compiler FIR is used.
         * This runner only works with Kotlin Language Version 2.0+ and is disabled by default.
         */
        @JvmField
        public val USE_FIR_RUNNER: Option<Boolean> = Option("USE_FIR_RUNNER")
    }
}
