/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.expressions.FirSmartCastExpression
import ksp.org.jetbrains.kotlin.fir.expressions.toReference
import ksp.org.jetbrains.kotlin.fir.isDelegated
import ksp.org.jetbrains.kotlin.fir.isEnabled
import ksp.org.jetbrains.kotlin.fir.references.toResolvedSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirPropertySymbol
import ksp.org.jetbrains.kotlin.fir.types.resolvedType

object FirDeprecatedSmartCastChecker : FirSmartCastExpressionChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirSmartCastExpression) {
        if (LanguageFeature.UnstableSmartcastOnDelegatedProperties.isEnabled()) return // No need to run this checker
        if (!expression.isStable) return // Unstable smartcasts are already errors.

        val source = expression.source ?: return
        val symbol = expression.originalExpression.toReference(context.session)?.toResolvedSymbol<FirPropertySymbol>() ?: return
        if (symbol.isDelegated) {
            reporter.reportOn(source, FirErrors.DEPRECATED_SMARTCAST_ON_DELEGATED_PROPERTY, expression.resolvedType, symbol)
        }
    }
}
