/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.jvm.checkers.expression

import ksp.org.jetbrains.kotlin.KtRealSourceElementKind
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirAnnotationChecker
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.jvm.FirJvmErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirDeclarationOrigin
import ksp.org.jetbrains.kotlin.fir.expressions.FirAnnotation
import ksp.org.jetbrains.kotlin.fir.expressions.FirAnnotationCall
import ksp.org.jetbrains.kotlin.fir.expressions.FirWrappedArgumentExpression
import ksp.org.jetbrains.kotlin.fir.expressions.impl.FirResolvedArgumentList
import ksp.org.jetbrains.kotlin.fir.resolve.toClassSymbol
import ksp.org.jetbrains.kotlin.fir.types.abbreviatedTypeOrSelf
import ksp.org.jetbrains.kotlin.fir.types.classLikeLookupTagIfAny
import ksp.org.jetbrains.kotlin.fir.types.coneType
import ksp.org.jetbrains.kotlin.name.ClassId
import ksp.org.jetbrains.kotlin.name.JvmStandardClassIds
import ksp.org.jetbrains.kotlin.name.StandardClassIds.Annotations

object FirJavaAnnotationsChecker : FirAnnotationChecker(MppCheckerKind.Common) {

    private val javaToKotlinNameMap: Map<ClassId, ClassId> =
        mapOf(
            JvmStandardClassIds.Annotations.Java.Target to Annotations.Target,
            JvmStandardClassIds.Annotations.Java.Retention to Annotations.Retention,
            JvmStandardClassIds.Annotations.Java.Deprecated to Annotations.Deprecated,
            JvmStandardClassIds.Annotations.Java.Documented to Annotations.MustBeDocumented,
        )

    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirAnnotation) {
        if (context.containingDeclarations.lastOrNull()?.source?.kind != KtRealSourceElementKind) return
        val annotationType = expression.annotationTypeRef.coneType.abbreviatedTypeOrSelf
        val classSymbol = annotationType.classLikeLookupTagIfAny?.toClassSymbol() ?: return
        if (classSymbol.origin !is FirDeclarationOrigin.Java) return

        val lookupTag = classSymbol.toLookupTag()
        javaToKotlinNameMap[lookupTag.classId]?.let { betterName ->
            reporter.reportOn(expression.source, FirJvmErrors.DEPRECATED_JAVA_ANNOTATION, betterName.asSingleFqName())
        }

        if (expression is FirAnnotationCall) {
            val argumentList = expression.argumentList
            if (argumentList is FirResolvedArgumentList) {
                val arguments = argumentList.originalArgumentList?.arguments ?: return
                for (key in arguments) {
                    if (key !is FirWrappedArgumentExpression && argumentList.mapping[key]?.name.let { it != null && it != Annotations.ParameterNames.value}) {
                        reporter.reportOn(key.source, FirJvmErrors.POSITIONED_VALUE_ARGUMENT_FOR_JAVA_ANNOTATION)
                    }
                }
            }
        }
    }
}
