/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.diagnostics

import ksp.com.intellij.lang.ASTNode
import ksp.com.intellij.openapi.util.TextRange
import ksp.com.intellij.psi.PsiComment
import ksp.com.intellij.psi.PsiElement
import ksp.com.intellij.psi.PsiErrorElement
import ksp.com.intellij.psi.PsiWhiteSpace
import ksp.org.jetbrains.kotlin.psi.psiUtil.endOffset
import ksp.org.jetbrains.kotlin.psi.psiUtil.startOffset

/**
 * Overriding `isValid` can lead to a loss of diagnostics in tests.
 *
 * Note that it's only called in tests. In the real world, the diagnostics are still reported.
 *
 * Instead of filtering diagnostics using `isValid`, the checker should be adapted.
 */
@RequiresOptIn
annotation class DiagnosticLossRisk

open class PositioningStrategy<in E : PsiElement> {
    open fun markDiagnostic(diagnostic: DiagnosticMarker): List<TextRange> {
        @Suppress("UNCHECKED_CAST")
        return mark(diagnostic.psiElement as E)
    }

    open fun mark(element: E): List<TextRange> {
        return markElement(element)
    }

    @DiagnosticLossRisk
    open fun isValid(element: E): Boolean {
        return !hasSyntaxErrors(element)
    }
}

fun markElement(element: PsiElement): List<TextRange> {
    return listOf(TextRange(getStartOffset(element), getEndOffset(element)))
}

fun markSingleElement(element: PsiElement): TextRange {
    return TextRange(getStartOffset(element), getEndOffset(element))
}

fun markNode(node: ASTNode): List<TextRange> {
    return markElement(node.psi)
}

fun markRange(range: TextRange): List<TextRange> {
    return listOf(range)
}

fun markRange(from: PsiElement, to: PsiElement): List<TextRange> {
    return markRange(TextRange(getStartOffset(from), getEndOffset(to)))
}

private fun getStartOffset(element: PsiElement): Int {
    var child = element.firstChild
    if (child != null) {
        while (child is PsiComment || child is PsiWhiteSpace) {
            child = child.nextSibling
        }
        if (child != null) {
            return getStartOffset(child)
        }
    }
    return element.startOffset
}

private fun getEndOffset(element: PsiElement): Int {
    var child = element.lastChild
    if (child != null) {
        while (child is PsiComment || child is PsiWhiteSpace) {
            child = child.prevSibling
        }
        if (child != null) {
            return getEndOffset(child)
        }
    }
    return element.endOffset
}

fun hasSyntaxErrors(psiElement: PsiElement): Boolean {
    if (psiElement is PsiErrorElement) return true

    val children = psiElement.children
    return children.isNotEmpty() && hasSyntaxErrors(children.last())
}

