/*
 * Copyright 2010-2022 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.session

import ksp.org.jetbrains.kotlin.config.CompilerConfiguration
import ksp.org.jetbrains.kotlin.fir.FirBinaryDependenciesModuleData
import ksp.org.jetbrains.kotlin.fir.FirPlatformSpecificCastChecker
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.SessionConfiguration
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.PlatformConflictDeclarationsDiagnosticDispatcher
import ksp.org.jetbrains.kotlin.fir.analysis.native.checkers.FirNativeCastChecker
import ksp.org.jetbrains.kotlin.fir.analysis.native.checkers.NativeConflictDeclarationsDiagnosticDispatcher
import ksp.org.jetbrains.kotlin.fir.backend.native.FirNativeClassMapper
import ksp.org.jetbrains.kotlin.fir.backend.native.FirNativeOverrideChecker
import ksp.org.jetbrains.kotlin.fir.checkers.registerExtraNativeCheckers
import ksp.org.jetbrains.kotlin.fir.checkers.registerNativeCheckers
import ksp.org.jetbrains.kotlin.fir.deserialization.ModuleDataProvider
import ksp.org.jetbrains.kotlin.fir.resolve.providers.FirSymbolProvider
import ksp.org.jetbrains.kotlin.fir.scopes.FirDefaultImportProviderHolder
import ksp.org.jetbrains.kotlin.fir.scopes.FirKotlinScopeProvider
import ksp.org.jetbrains.kotlin.fir.scopes.FirOverrideChecker
import ksp.org.jetbrains.kotlin.fir.scopes.FirPlatformClassMapper
import ksp.org.jetbrains.kotlin.library.KotlinLibrary
import ksp.org.jetbrains.kotlin.library.metadata.impl.KlibResolvedModuleDescriptorsFactoryImpl.Companion.FORWARD_DECLARATIONS_MODULE_NAME
import ksp.org.jetbrains.kotlin.resolve.konan.platform.NativePlatformAnalyzerServices

@OptIn(SessionConfiguration::class)
abstract class FirNativeSessionFactory : AbstractFirKlibSessionFactory<Nothing?, Nothing?>() {
    companion object : FirNativeSessionFactory()

    object ForMetadata : FirNativeSessionFactory() {
        override val requiresSpecialSetupOfSourceProvidersInHmppCompilation: Boolean
            get() = false
    }


    // ==================================== Library session ====================================

    override fun createLibraryContext(configuration: CompilerConfiguration): Nothing? {
        return null
    }

    override fun createAdditionalDependencyProviders(
        session: FirSession,
        moduleDataProvider: ModuleDataProvider,
        kotlinScopeProvider: FirKotlinScopeProvider,
        resolvedLibraries: List<KotlinLibrary>,
    ): List<FirSymbolProvider> {
        val forwardDeclarationsModuleData =
            FirBinaryDependenciesModuleData(FORWARD_DECLARATIONS_MODULE_NAME).apply {
                bindSession(session)
            }
        val provider = NativeForwardDeclarationsSymbolProvider(
            session,
            forwardDeclarationsModuleData,
            kotlinScopeProvider,
            resolvedLibraries
        )
        return listOf(provider)
    }

    override fun FirSession.registerLibrarySessionComponents(c: Nothing?) {
        registerComponents()
    }

    // ==================================== Platform session ====================================

    override fun FirSessionConfigurator.registerPlatformCheckers(c: Nothing?) {
        registerNativeCheckers()
    }

    override fun FirSessionConfigurator.registerExtraPlatformCheckers(c: Nothing?) {
        registerExtraNativeCheckers()
    }

    override fun FirSession.registerSourceSessionComponents(c: Nothing?) {
        registerComponents()
    }

    override fun createSourceContext(configuration: CompilerConfiguration): Nothing? {
        return null
    }

    // ==================================== Common parts ====================================

    private fun FirSession.registerComponents() {
        registerDefaultComponents()
        registerNativeComponents()
    }

    // ==================================== Utilities ====================================

    fun FirSession.registerNativeComponents() {
        register(FirPlatformClassMapper::class, FirNativeClassMapper())
        register(FirPlatformSpecificCastChecker::class, FirNativeCastChecker)
        register(PlatformConflictDeclarationsDiagnosticDispatcher::class, NativeConflictDeclarationsDiagnosticDispatcher)
        register(FirOverrideChecker::class, FirNativeOverrideChecker(this))
        register(FirDefaultImportProviderHolder::class, FirDefaultImportProviderHolder(NativePlatformAnalyzerServices))
    }
}
