/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.psi.stubs.impl

import ksp.com.intellij.psi.PsiElement
import ksp.com.intellij.psi.stubs.StubElement
import ksp.com.intellij.psi.stubs.StubInputStream
import ksp.com.intellij.psi.stubs.StubOutputStream
import ksp.com.intellij.util.io.StringRef
import ksp.org.jetbrains.kotlin.contracts.description.KtContractDescriptionElement
import ksp.org.jetbrains.kotlin.name.FqName
import ksp.org.jetbrains.kotlin.psi.KtNamedFunction
import ksp.org.jetbrains.kotlin.psi.stubs.KotlinFunctionStub
import ksp.org.jetbrains.kotlin.psi.stubs.elements.KtStubElementTypes
import java.io.IOException

class KotlinFunctionStubImpl(
    parent: StubElement<out PsiElement>?,
    private val nameRef: StringRef?,
    override val isTopLevel: Boolean,
    override val fqName: FqName?,
    override val isExtension: Boolean,
    override val hasNoExpressionBody: Boolean,
    override val hasBody: Boolean,
    override val hasTypeParameterListBeforeFunctionName: Boolean,
    override val mayHaveContract: Boolean,
    val contract: List<KtContractDescriptionElement<KotlinTypeBean, Nothing?>>?,
    val origin: KotlinStubOrigin?
) : KotlinStubBaseImpl<KtNamedFunction>(parent, KtStubElementTypes.FUNCTION), KotlinFunctionStub {
    init {
        if (isTopLevel && fqName == null) {
            throw IllegalArgumentException("fqName shouldn't be null for top level functions")
        }
    }

    override fun getName() = StringRef.toString(nameRef)

    @Throws(IOException::class)
    fun serializeContract(dataStream: StubOutputStream) {
        val effects: List<KtContractDescriptionElement<KotlinTypeBean, Nothing?>>? = contract
        dataStream.writeVarInt(effects?.size ?: 0)
        val visitor = KotlinContractSerializationVisitor(dataStream)
        effects?.forEach { it.accept(visitor, null) }
    }

    companion object {
        fun deserializeContract(dataStream: StubInputStream): List<KtContractDescriptionElement<KotlinTypeBean, Nothing?>> {
            val effects = mutableListOf<KtContractDescriptionElement<KotlinTypeBean, Nothing?>>()
            val count: Int = dataStream.readVarInt()
            for (i in 0 until count) {
                val effectType: KotlinContractEffectType = KotlinContractEffectType.entries[dataStream.readVarInt()]
                effects.add(effectType.deserialize(dataStream))
            }
            return effects
        }
    }
}
