/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.jvm.checkers.expression

import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.isValueClass
import ksp.org.jetbrains.kotlin.fir.declarations.hasAnnotation
import ksp.org.jetbrains.kotlin.fir.languageVersionSettings
import ksp.org.jetbrains.kotlin.fir.resolve.toClassSymbol
import ksp.org.jetbrains.kotlin.fir.types.ConeFlexibleType
import ksp.org.jetbrains.kotlin.fir.types.ConeKotlinType
import ksp.org.jetbrains.kotlin.fir.types.isPrimitiveOrNullablePrimitive
import ksp.org.jetbrains.kotlin.name.ClassId

private val jdkInternalValueBasedAnnotationClassId = ClassId.fromString("jdk/internal/ValueBased")

internal fun ConeKotlinType.isJavaValueBasedClass(session: FirSession): Boolean {
    val classSymbol = toClassSymbol(session) ?: return false
    return classSymbol.hasAnnotation(jdkInternalValueBasedAnnotationClassId, session)
}

internal fun ConeKotlinType.isJavaValueBasedClassAndWarningsEnabled(session: FirSession): Boolean {
    return !session.languageVersionSettings.supportsFeature(LanguageFeature.DisableWarningsForValueBasedJavaClasses)
            && this.isJavaValueBasedClass(session)
}

internal fun ConeKotlinType.isValueTypeAndWarningsEnabled(session: FirSession): Boolean {
    if (!session.languageVersionSettings.supportsFeature(LanguageFeature.DisableWarningsForIdentitySensitiveOperationsOnValueClassesAndPrimitives) &&
        (this.isPrimitiveOrNullablePrimitive || this.isValueClass(session) || this.isFlexiblePrimitive())
    ) return true
    return this.isJavaValueBasedClassAndWarningsEnabled(session)
}

internal fun ConeKotlinType.isFlexiblePrimitive(): Boolean {
    return this is ConeFlexibleType && lowerBound.isPrimitiveOrNullablePrimitive && upperBound.isPrimitiveOrNullablePrimitive
}
