/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.impl.base.util

import ksp.com.intellij.psi.util.PsiTreeUtil
import ksp.org.jetbrains.kotlin.analysis.api.KaImplementationDetail
import ksp.org.jetbrains.kotlin.name.CallableId
import ksp.org.jetbrains.kotlin.name.Name
import ksp.org.jetbrains.kotlin.psi.KtCallableDeclaration
import ksp.org.jetbrains.kotlin.psi.KtClassOrObject
import ksp.org.jetbrains.kotlin.psi.KtCodeFragment
import ksp.org.jetbrains.kotlin.psi.KtDeclaration
import ksp.org.jetbrains.kotlin.psi.KtEnumEntry
import ksp.org.jetbrains.kotlin.psi.KtFile
import ksp.org.jetbrains.kotlin.psi.KtNamedFunction
import ksp.org.jetbrains.kotlin.psi.KtProperty
import ksp.org.jetbrains.kotlin.psi.psiUtil.containingClassOrObject

@KaImplementationDetail
val KtCallableDeclaration.callableId: CallableId?
    get() = when (this) {
        is KtNamedFunction -> callableId
        is KtProperty -> callableId
        else -> null
    }

/**
 * Not null [CallableId] for functions which are not local and are not a member of a local class.
 */
@KaImplementationDetail
val KtNamedFunction.callableId: CallableId?
    get() = if (isLocal) null else callableIdForName(nameAsSafeName)

@KaImplementationDetail
val KtEnumEntry.callableId: CallableId?
    get() = callableIdForName(nameAsSafeName)

@KaImplementationDetail
val KtProperty.callableId: CallableId?
    get() = if (isLocal) null else callableIdForName(nameAsSafeName)

@KaImplementationDetail
fun KtDeclaration.callableIdForName(callableName: Name): CallableId? {
    val containingClassOrObject = containingClassOrObject
    if (containingClassOrObject != null) {
        return containingClassOrObject.getClassId()?.let { classId ->
            CallableId(classId = classId, callableName = callableName)
        }
    }

    return CallableId(packageName = containingKtFile.packageFqName, callableName = callableName)
}