/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.js.checkers.expression

import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirCallChecker
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.js.FirJsErrors
import ksp.org.jetbrains.kotlin.fir.declarations.hasAnnotation
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isEffectivelyExternal
import ksp.org.jetbrains.kotlin.fir.expressions.FirCall
import ksp.org.jetbrains.kotlin.fir.expressions.resolvedArgumentMapping
import ksp.org.jetbrains.kotlin.fir.expressions.unwrapArgument
import ksp.org.jetbrains.kotlin.fir.types.ConeDynamicType
import ksp.org.jetbrains.kotlin.fir.types.resolvedType
import ksp.org.jetbrains.kotlin.fir.resolve.toRegularClassSymbol
import ksp.org.jetbrains.kotlin.name.JsStandardClassIds.Annotations.JsExternalArgument

object FirJsExternalArgumentCallChecker : FirCallChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirCall) {
        val arguments = expression.resolvedArgumentMapping ?: return
        for ((argument, parameter) in arguments) {
            if (parameter.hasAnnotation(JsExternalArgument, context.session)) {
                val unwrappedArg = argument.unwrapArgument()
                val type = unwrappedArg.resolvedType
                val symbol = type.toRegularClassSymbol(context.session)
                if (symbol?.isEffectivelyExternal(context.session) == false || type is ConeDynamicType) {
                    reporter.reportOn(
                        unwrappedArg.source,
                        FirJsErrors.JS_EXTERNAL_ARGUMENT,
                        type
                    )
                }
            }
        }
    }
}
