/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.native.checkers

import ksp.org.jetbrains.kotlin.KtRealSourceElementKind
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.FirCallableDeclarationChecker
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.fullyExpandedClassId
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.native.FirNativeErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirAnonymousFunction
import ksp.org.jetbrains.kotlin.fir.declarations.FirCallableDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.FirFunction
import ksp.org.jetbrains.kotlin.fir.declarations.FirPropertyAccessor
import ksp.org.jetbrains.kotlin.fir.declarations.FirValueParameter
import ksp.org.jetbrains.kotlin.fir.declarations.utils.visibility
import ksp.org.jetbrains.kotlin.fir.types.FirTypeRef
import ksp.org.jetbrains.kotlin.fir.types.coneType
import ksp.org.jetbrains.kotlin.name.FqName
import ksp.org.jetbrains.kotlin.name.Name

object FirNativeSpecificAtomicChecker : FirCallableDeclarationChecker(MppCheckerKind.Platform) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirCallableDeclaration) {
        if (!declaration.visibility.isPublicAPI || declaration is FirValueParameter || declaration is FirAnonymousFunction) return
        declaration.receiverParameter?.typeRef?.let {
            checkType(it)
        }
        declaration.contextParameters.forEach {
            checkType(it.returnTypeRef)
        }
        declaration.returnTypeRef.takeIf { it.source?.kind is KtRealSourceElementKind }?.let {
            checkType(it)
        }
        // Note: not much sense to check type parameter bounds, or class supertypes: all atomics are final types
        if (declaration !is FirFunction || declaration is FirPropertyAccessor) return
        declaration.valueParameters.forEach {
            checkType(it.returnTypeRef)
        }
    }

    context(context: CheckerContext, reporter: DiagnosticReporter)
    private fun checkType(
        typeRef: FirTypeRef,
    ) {
        val classId = typeRef.coneType.fullyExpandedClassId(context.session) ?: return
        if (classId.packageFqName != CONCURRENT_PACKAGE) return
        if (classId.parentClassId != null) return
        val name = classId.shortClassName
        if (name !in CONCURRENT_NAME_SET) return
        reporter.reportOn(typeRef.source, FirNativeErrors.NATIVE_SPECIFIC_ATOMIC, name)
    }

    private val CONCURRENT_PACKAGE = FqName("kotlin.concurrent")
    private val CONCURRENT_NAME_SET = listOf(
        "AtomicIntArray",
        "AtomicLongArray",
        "AtomicArray",
        "AtomicInt",
        "AtomicLong",
        "AtomicReference",
    ).mapTo(mutableSetOf()) { Name.identifier(it) }.toSet()
}
