/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.scopes.impl

import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isStatic
import ksp.org.jetbrains.kotlin.fir.resolve.ScopeSession
import ksp.org.jetbrains.kotlin.fir.resolve.substitution.ConeSubstitutor
import ksp.org.jetbrains.kotlin.fir.scopes.DelicateScopeAPI
import ksp.org.jetbrains.kotlin.fir.scopes.FirContainingNamesAwareScope
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirClassifierSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirNamedFunctionSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirVariableSymbol
import ksp.org.jetbrains.kotlin.name.Name

class FirStaticScope(private val delegateScope: FirContainingNamesAwareScope) : FirContainingNamesAwareScope() {
    // We want to *avoid* delegation to certain scope functions, so we delegate explicitly instead of using
    // `FirDelegatingContainingNamesAwareScope`.

    override fun processClassifiersByNameWithSubstitution(name: Name, processor: (FirClassifierSymbol<*>, ConeSubstitutor) -> Unit) {
        delegateScope.processClassifiersByNameWithSubstitution(name, processor)
    }

    override fun processFunctionsByName(name: Name, processor: (FirNamedFunctionSymbol) -> Unit) {
        delegateScope.processFunctionsByName(name) {
            if (it.fir.isStatic) {
                processor(it)
            }
        }
    }

    override fun processPropertiesByName(name: Name, processor: (FirVariableSymbol<*>) -> Unit) {
        delegateScope.processPropertiesByName(name) {
            if (it.fir.isStatic) {
                processor(it)
            }
        }
    }


    override fun getCallableNames(): Set<Name> {
        return delegateScope.getCallableNames()
    }

    override fun getClassifierNames(): Set<Name> {
        return delegateScope.getClassifierNames()
    }

    override val scopeOwnerLookupNames: List<String>
        get() = delegateScope.scopeOwnerLookupNames

    @DelicateScopeAPI
    override fun withReplacedSessionOrNull(newSession: FirSession, newScopeSession: ScopeSession): FirStaticScope? {
        return delegateScope.withReplacedSessionOrNull(newSession, newScopeSession)?.let { FirStaticScope(it) }
    }
}
