// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package ksp.com.intellij.patterns;

import ksp.com.intellij.psi.PsiAnnotation;
import ksp.com.intellij.psi.PsiArrayInitializerMemberValue;
import ksp.com.intellij.psi.PsiElement;
import ksp.com.intellij.util.ProcessingContext;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;

import static com.intellij.patterns.PsiJavaPatterns.psiNameValuePair;

public final class PsiAnnotationPattern extends PsiElementPattern<PsiAnnotation, PsiAnnotationPattern> {
  static final PsiAnnotationPattern PSI_ANNOTATION_PATTERN = new PsiAnnotationPattern();

  private PsiAnnotationPattern() {
    super(PsiAnnotation.class);
  }

  public PsiAnnotationPattern qName(final ElementPattern<String> pattern) {
    return with(new PatternCondition<PsiAnnotation>("qName") {
      @Override
      public boolean accepts(@NotNull final PsiAnnotation psiAnnotation, final ProcessingContext context) {
        return pattern.accepts(psiAnnotation.getQualifiedName(), context);
      }
    });
  }
  public PsiAnnotationPattern qName(@NonNls final String qname) {
    return with(new PatternCondition<PsiAnnotation>("qName") {
      @Override
      public boolean accepts(@NotNull final PsiAnnotation psiAnnotation, final ProcessingContext context) {
        return psiAnnotation.hasQualifiedName(qname);
      }
    });
  }

  public PsiAnnotationPattern insideAnnotationAttribute(@NotNull final String attributeName, @NotNull final ElementPattern<? extends PsiAnnotation> parentAnnoPattern) {
    return with(new PatternCondition<PsiAnnotation>("insideAnnotationAttribute") {
      final PsiNameValuePairPattern attrPattern = psiNameValuePair().withName(attributeName).withSuperParent(2, parentAnnoPattern);

      @Override
      public boolean accepts(@NotNull PsiAnnotation annotation, ProcessingContext context) {
        PsiElement attr = getParentElement(annotation);
        if (attr instanceof PsiArrayInitializerMemberValue) attr = getParentElement(attr);
        return attrPattern.accepts(attr);
      }
    });
  }

  private PsiElement getParentElement(@NotNull PsiElement element) {
    return getParent(element);
  }
}
