// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package ksp.com.intellij.patterns;

import ksp.com.intellij.codeInsight.AnnotationUtil;
import ksp.com.intellij.psi.PsiModifierList;
import ksp.com.intellij.psi.PsiModifierListOwner;
import ksp.com.intellij.util.ProcessingContext;
import ksp.com.intellij.util.containers.ContainerUtil;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;

public class PsiModifierListOwnerPattern<T extends PsiModifierListOwner, Self extends PsiModifierListOwnerPattern<T,Self>> extends PsiElementPattern<T,Self> {
  public PsiModifierListOwnerPattern(@NotNull final InitialPatternCondition<T> condition) {
    super(condition);
  }

  protected PsiModifierListOwnerPattern(final Class<T> aClass) {
    super(aClass);
  }

  public Self withModifiers(final String... modifiers) {
    return with(new PatternCondition<T>("withModifiers") {
      @Override
      public boolean accepts(@NotNull final T t, final ProcessingContext context) {
        return ContainerUtil.and(modifiers, s -> t.hasModifierProperty(s));
      }
    });
  }

  public Self withoutModifiers(final String... modifiers) {
    return with(new PatternCondition<T>("withoutModifiers") {
      @Override
      public boolean accepts(@NotNull final T t, final ProcessingContext context) {
        return ContainerUtil.and(modifiers, s -> !t.hasModifierProperty(s));
      }
    });
  }

  public Self withAnnotation(@NonNls final String qualifiedName) {
    return with(new PatternCondition<T>("withAnnotation") {
      @Override
      public boolean accepts(@NotNull final T t, final ProcessingContext context) {
        final PsiModifierList modifierList = t.getModifierList();
        return modifierList != null && modifierList.hasAnnotation(qualifiedName);
      }
    });
  }

  public Self withAnnotations(@NonNls final String... qualifiedNames) {
    return with(new PatternCondition<T>("withAnnotations") {
      @Override
      public boolean accepts(@NotNull final T t, final ProcessingContext context) {
        return AnnotationUtil.findAnnotation(t, qualifiedNames) != null;
      }
    });
  }

  public static class Capture<T extends PsiModifierListOwner> extends PsiModifierListOwnerPattern<T, Capture<T>> {
    public Capture(@NotNull InitialPatternCondition<T> condition) {
      super(condition);
    }
  }
}
