// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.psi;

import ksp.com.intellij.lang.ASTNode;
import ksp.com.intellij.lang.FileASTNode;
import ksp.com.intellij.lang.Language;
import ksp.com.intellij.openapi.diagnostic.Attachment;
import ksp.com.intellij.openapi.diagnostic.ExceptionWithAttachments;
import ksp.com.intellij.openapi.util.Key;
import ksp.com.intellij.openapi.util.UserDataHolder;
import ksp.com.intellij.openapi.util.registry.Registry;
import ksp.com.intellij.openapi.vfs.VirtualFile;
import ksp.com.intellij.psi.stubs.PsiFileStub;
import ksp.com.intellij.psi.stubs.StubElement;
import ksp.com.intellij.psi.tree.IElementType;
import ksp.com.intellij.psi.util.PsiUtilCore;
import ksp.com.intellij.util.containers.JBIterable;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

import java.lang.ref.SoftReference;

public final class PsiInvalidElementAccessException extends RuntimeException implements ExceptionWithAttachments {
  private static final Key<Object> INVALIDATION_TRACE = Key.create("INVALIDATION_TRACE");
  private static final Key<Boolean> REPORTING_EXCEPTION = Key.create("REPORTING_EXCEPTION");

  private final SoftReference<PsiElement> myElementReference;  // to prevent leaks, since exceptions are stored in IdeaLogger
  private final Attachment[] myDiagnostic;
  private final @NonNls String myMessage;

  public PsiInvalidElementAccessException(@Nullable PsiElement element) {
    this(element, null, null);
  }

  public PsiInvalidElementAccessException(@Nullable PsiElement element, @Nullable @NonNls String message) {
    this(element, message, null);
  }

  public PsiInvalidElementAccessException(@Nullable PsiElement element, @Nullable Throwable cause) {
    this(element, null, cause);
  }

  public PsiInvalidElementAccessException(@Nullable PsiElement element, @Nullable @NonNls String message, @Nullable Throwable cause) {
    super(null, cause);
    myElementReference = new SoftReference<>(element);

    if (element == null) {
      myMessage = message;
      myDiagnostic = Attachment.EMPTY_ARRAY;
    }
    else if (element == PsiUtilCore.NULL_PSI_ELEMENT) {
      myMessage = "NULL_PSI_ELEMENT ;" + message;
      myDiagnostic = Attachment.EMPTY_ARRAY;
    }
    else {
      boolean recursiveInvocation = Boolean.TRUE.equals(element.getUserData(REPORTING_EXCEPTION));
      element.putUserData(REPORTING_EXCEPTION, Boolean.TRUE);

      try {
        Object trace = recursiveInvocation ? null : getPsiInvalidationTrace(element);
        myMessage = getMessageWithReason(element, message, recursiveInvocation, trace);
        myDiagnostic = createAttachments(trace);
      }
      finally {
        element.putUserData(REPORTING_EXCEPTION, null);
      }
    }
  }

  private PsiInvalidElementAccessException(@NotNull ASTNode node, @Nullable @NonNls String message) {
    myElementReference = new SoftReference<>(null);
    final IElementType elementType = node.getElementType();
    myMessage = "Element " + node.getClass() + " of type " + elementType + " (" + elementType.getClass() + ")" +
                (message == null ? "" : "; " + message);
    myDiagnostic = createAttachments(findInvalidationTrace(node));
  }

  @NotNull
  public static PsiInvalidElementAccessException createByNode(@NotNull ASTNode node, @Nullable @NonNls String message) {
    return new PsiInvalidElementAccessException(node, message);
  }

  private static Attachment @NotNull [] createAttachments(@Nullable Object trace) {
    return trace == null
           ? Attachment.EMPTY_ARRAY
           : new Attachment[]{trace instanceof Throwable ? new Attachment("invalidation", (Throwable)trace)
                                                         : new Attachment("diagnostic.txt", trace.toString())};
  }

  private static @Nullable Object getPsiInvalidationTrace(@NotNull PsiElement element) {
    Object trace = getInvalidationTrace(element);
    if (trace != null) return trace;

    if (element instanceof PsiFile) {
      return getInvalidationTrace(((PsiFile)element).getOriginalFile());
    }
    return findInvalidationTrace(element.getNode());
  }

  private static @NotNull String getMessageWithReason(@NotNull PsiElement element,
                                                      @Nullable String message,
                                                      boolean recursiveInvocation,
                                                      @Nullable Object trace) {
    @NonNls String reason = "Element: " + element.getClass();
    if (!recursiveInvocation) {
      try {
        reason += " #" + getLanguage(element).getID() + " ";
      }
      catch (PsiInvalidElementAccessException ignore) { }
      String traceText = !isTrackingInvalidation() ? "disabled" :
                         trace != null ? "see attachment" :
                         "no info";
      try {
        reason += " because: " + findOutInvalidationReason(element);
      }
      catch (PsiInvalidElementAccessException ignore) { }
      reason += "\ninvalidated at: " + traceText;
    }
    return reason + (message == null ? "" : "; " + message);
  }

  private static @NotNull Language getLanguage(@NotNull PsiElement element) {
    return element instanceof ASTNode ? ((ASTNode)element).getElementType().getLanguage() : element.getLanguage();
  }

  @Override
  public String getMessage() {
    return myMessage;
  }

  @Override
  public Attachment @NotNull [] getAttachments() {
    return myDiagnostic;
  }

  public static Object findInvalidationTrace(@Nullable ASTNode element) {
    while (element != null) {
      Object trace = element.getUserData(INVALIDATION_TRACE);
      if (trace != null) {
        return trace;
      }
      ASTNode parent = element.getTreeParent();
      if (parent == null && element instanceof FileASTNode) {
        PsiElement psi = element.getPsi();
        trace = psi == null ? null : psi.getUserData(INVALIDATION_TRACE);
        if (trace != null) {
          return trace;
        }
      }
      element = parent;
    }
    return null;
  }

  public static @NonNls @NotNull String findOutInvalidationReason(@NotNull PsiElement root) {
    if (root == PsiUtilCore.NULL_PSI_ELEMENT) {
      return "NULL_PSI_ELEMENT";
    }

    PsiElement lastParent = root;
    PsiElement element = root instanceof PsiFile ? root : root.getParent();
    if (element == null) {
      @NonNls String m = "parent is null";
      if (root instanceof StubBasedPsiElement) {
        StubElement<?> stub = ((StubBasedPsiElement<?>)root).getStub();
        while (stub != null) {
          //noinspection StringConcatenationInLoop
          m += "\n  each stub=" + stub;
          if (stub instanceof PsiFileStub) {
            m += "; fileStub.psi=" + stub.getPsi() + "; reason=" + ((PsiFileStub<?>)stub).getInvalidationReason();
          }
          stub = stub.getParentStub();
        }
      }
      return m;
    }

    String hierarchy = "";
    while (element != null && !(element instanceof PsiFile)) {
      //noinspection StringConcatenationInLoop
      hierarchy += (hierarchy.isEmpty() ? "" : ", ") + element.getClass();
      lastParent = element;
      element = element.getParent();
    }
    PsiFile psiFile = (PsiFile)element;
    if (psiFile == null) {
      PsiElement context = lastParent.getContext();
      return "containing file is null; hierarchy=" + hierarchy +
             ", context=" + context +
             ", contextFile=" + JBIterable.generate(context, PsiElement::getParent).find(e -> e instanceof PsiFile);
    }

    FileViewProvider provider = psiFile.getViewProvider();
    VirtualFile vFile = provider.getVirtualFile();
    if (!vFile.isValid()) {
      return vFile + " is invalid";
    }
    if (!provider.isPhysical()) {
      PsiElement context = psiFile.getContext();
      if (context != null && !context.isValid()) {
        return "invalid context: " + findOutInvalidationReason(context);
      }
    }

    PsiFile original = psiFile.getOriginalFile();
    if (original != psiFile && !original.isValid()) {
      return "invalid original: " + findOutInvalidationReason(original);
    }

    PsiManager manager = psiFile.getManager();
    if (manager.getProject().isDisposed()) {
      return "project is disposed: " + manager.getProject();
    }

    Language language = psiFile.getLanguage();
    if (language != provider.getBaseLanguage()) {
      return "File language:" + language + " != Provider base language:" + provider.getBaseLanguage();
    }

    FileViewProvider p = manager.findCachedViewProvider(vFile);
    if (provider != p) {
      return "different providers: " + provider + "(" + id(provider) + "); " + p + "(" + id(p) + ")";
    }

    if (!provider.isPhysical()) {
      return "non-physical provider: " + provider; // "dummy" file?
    }

    return "psi is outdated";
  }

  private static @NotNull String id(@Nullable FileViewProvider provider) {
    return Integer.toHexString(System.identityHashCode(provider));
  }

  public static void setInvalidationTrace(@NotNull UserDataHolder element, Object trace) {
    element.putUserData(INVALIDATION_TRACE, trace);
  }

  public static Object getInvalidationTrace(@NotNull UserDataHolder element) {
    return element.getUserData(INVALIDATION_TRACE);
  }

  public static boolean isTrackingInvalidation() {
    return Registry.is("psi.track.invalidation", true);
  }

  public @Nullable PsiElement getPsiElement() {
    return myElementReference.get();
  }
}
