// Copyright 2000-2022 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.psi;

import ksp.com.intellij.lang.jvm.JvmParameter;
import ksp.com.intellij.openapi.util.NlsSafe;
import ksp.com.intellij.util.ArrayFactory;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

/**
 * Represents the parameter of a Java method, foreach (enhanced for) statement or catch block.
 */
public interface PsiParameter extends PsiVariable, JvmParameter, PsiJvmModifiersOwner {
  /**
   * The empty array of PSI parameters which can be reused to avoid unnecessary allocations.
   */
  PsiParameter[] EMPTY_ARRAY = new PsiParameter[0];

  ArrayFactory<PsiParameter> ARRAY_FACTORY = count -> count == 0 ? EMPTY_ARRAY : new PsiParameter[count];

  /**
   * Returns the element (method, lambda expression, foreach statement or catch block) in which the
   * parameter is declared.
   *
   * @return the declaration scope for the parameter.
   */
  @NotNull PsiElement getDeclarationScope();

  /**
   * Checks if the parameter accepts a variable number of arguments.
   *
   * @return true if the parameter is a vararg, false otherwise
   */
  boolean isVarArgs();

  /**
   * {@inheritDoc}
   */
  @Override
  @Nullable PsiTypeElement getTypeElement();

  // This explicit declaration is required to force javac to generate a bridge method 'JvmType getType()'; without it calling
  // JvmParameter#getType() method on instances which weren't recompiled against the new API will cause AbstractMethodError.
  @Override
  @NotNull PsiType getType();

  // binary compatibility
  @Override
  default PsiAnnotation @NotNull [] getAnnotations() {
    return PsiJvmModifiersOwner.super.getAnnotations();
  }

  @Override
  @NotNull @NlsSafe String getName();
}
