// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.psi.impl;

import ksp.com.intellij.diagnostic.PluginException;
import ksp.com.intellij.lang.Language;
import ksp.com.intellij.openapi.project.Project;
import ksp.com.intellij.openapi.util.TextRange;
import ksp.com.intellij.psi.PsiElement;
import ksp.com.intellij.psi.PsiFile;
import ksp.com.intellij.psi.PsiManager;
import ksp.com.intellij.psi.meta.PsiMetaData;
import ksp.com.intellij.psi.meta.PsiMetaOwner;
import ksp.com.intellij.psi.meta.PsiPresentableMetaData;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

import javax.swing.*;

public abstract class RenameableFakePsiElement extends FakePsiElement implements PsiMetaOwner, PsiPresentableMetaData {
  private final PsiElement myParent;

  protected RenameableFakePsiElement(@Nullable PsiElement parent) {
    myParent = parent;
  }

  @Override
  public PsiElement getParent() {
    return myParent;
  }

  private @NotNull PsiElement getParentNotNull(@NotNull String methodName) {
    if (myParent != null) {
      return myParent;
    }
    throw PluginException.createByClass(
      new AbstractMethodError("Elements initialized with `null` parent are expected to override `#" + methodName + "`"),
      getClass()
    );
  }

  @Override
  public PsiFile getContainingFile() {
    return getParentNotNull("getContainingFile").getContainingFile();
  }

  @Override
  public abstract String getName();

  @Override
  public @NotNull Language getLanguage() {
    return getContainingFile().getLanguage();
  }

  @Override
  public @NotNull Project getProject() {
    return getParentNotNull("getProject").getProject();
  }

  @Override
  public PsiManager getManager() {
    return PsiManager.getInstance(getProject());
  }

  @Override
  public @Nullable PsiMetaData getMetaData() {
    return this;
  }

  @Override
  public PsiElement getDeclaration() {
    return this;
  }

  @Override
  public @NonNls String getName(PsiElement context) {
    return getName();
  }

  @Override
  public void init(PsiElement element) {
  }

  @Override
  public final @Nullable Icon getIcon(boolean open) {
    return getIcon();
  }

  @Override
  public @Nullable TextRange getTextRange() {
    return TextRange.from(0, 0);
  }
}
