// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package ksp.com.intellij.psi.impl.compiled;

import ksp.com.intellij.openapi.util.NotNullLazyValue;
import ksp.com.intellij.psi.PsiDocCommentOwner;
import ksp.com.intellij.psi.PsiElement;
import ksp.com.intellij.psi.PsiIdentifier;
import ksp.com.intellij.psi.PsiNameIdentifierOwner;
import ksp.com.intellij.psi.impl.PsiImplUtil;
import ksp.com.intellij.psi.impl.java.stubs.PsiMemberStub;
import ksp.com.intellij.psi.javadoc.PsiDocComment;
import ksp.com.intellij.util.IncorrectOperationException;
import ksp.org.jetbrains.annotations.NotNull;

public abstract class ClsMemberImpl<T extends PsiMemberStub> extends ClsRepositoryPsiElement<T> implements PsiDocCommentOwner, PsiNameIdentifierOwner {
  private final NotNullLazyValue<PsiDocComment> myDocComment;
  private final NotNullLazyValue<PsiIdentifier> myNameIdentifier;

  protected ClsMemberImpl(T stub) {
    super(stub);
    myDocComment = !stub.isDeprecated() ? null : NotNullLazyValue.atomicLazy(() -> {
      return new ClsDocCommentImpl(ClsMemberImpl.this);
    });
    myNameIdentifier = NotNullLazyValue.atomicLazy(() -> {
      return new ClsIdentifierImpl(ClsMemberImpl.this, getName());
    });
  }

  @Override
  public PsiDocComment getDocComment() {
    return myDocComment != null ? myDocComment.getValue() : null;
  }

  @Override
  @NotNull
  public PsiIdentifier getNameIdentifier() {
    return myNameIdentifier.getValue();
  }

  @Override
  @NotNull
  public String getName() {
    //noinspection ConstantConditions
    return getStub().getName();
  }

  @Override
  public PsiElement setName(@NotNull String name) throws IncorrectOperationException {
    PsiImplUtil.setName(getNameIdentifier(), name);
    return this;
  }
}
