// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.

package ksp.com.intellij.psi.tree;

import ksp.com.intellij.lang.Language;
import ksp.com.intellij.openapi.vfs.VirtualFile;
import ksp.com.intellij.psi.StubBuilder;
import ksp.com.intellij.psi.stubs.*;
import ksp.com.intellij.psi.templateLanguages.TemplateLanguage;
import ksp.com.intellij.util.ReflectionUtil;
import ksp.com.intellij.util.concurrency.SynchronizedClearableLazy;
import ksp.org.jetbrains.annotations.ApiStatus;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.util.Arrays;

public class IStubFileElementType<T extends PsiFileStub> extends StubFileElementType<T> {
  private static final SynchronizedClearableLazy<Integer> TEMPLATE_STUB_BASE_VERSION =
    new SynchronizedClearableLazy<>(IStubFileElementType::calcTemplateStubBaseVersion);

  public IStubFileElementType(Language language) {
    super(language);
  }

  public IStubFileElementType(@NonNls String debugName, Language language) {
    super(debugName, language);
    if (hasNonTrivialExternalId() && !isOutOfOurControl()) {
      IStubElementType.checkNotInstantiatedTooLate(getClass());
    }
    dropTemplateStubBaseVersion();
  }

  private boolean hasNonTrivialExternalId() {
    return ReflectionUtil.getMethodDeclaringClass(getClass(), "getExternalId") != IStubFileElementType.class;
  }

  private boolean isOutOfOurControl() {
    return getClass().getName().contains(".kotlin."); // KT-28732
  }

  /**
   * Stub structure version. Should be incremented each time when stub tree changes (e.g. elements added/removed,
   * element serialization/deserialization changes).
   * <p>
   * Make sure to invoke super method for {@link TemplateLanguage} to prevent stub serialization problems due to
   * data language stub changes.
   * <p>
   * Important: Negative values are not allowed! The platform relies on the fact that template languages have stub versions bigger than
   * {@link IStubFileElementType#TEMPLATE_STUB_BASE_VERSION}, see {@link StubBuilderType#getVersion()}. At the same time
   * {@link IStubFileElementType#TEMPLATE_STUB_BASE_VERSION} is computed as a sum of stub versions of all non-template languages.
   *
   * @return stub version
   */
  public int getStubVersion() {
    return getLanguage() instanceof TemplateLanguage ? TEMPLATE_STUB_BASE_VERSION.getValue() : 0;
  }

  public StubBuilder getBuilder() {
    return new DefaultStubBuilder();
  }

  /**
   * Can only depend on getLanguage() or getDebugName(), should be calculated inplace.
   * MUST NOT be calculated from any other non-static local variables as it can lead to race-conditions
   * (<a href="https://youtrack.jetbrains.com/issue/IDEA-306646">IDEA-306646</a>).
   */
  @Override
  public @NonNls @NotNull String getExternalId() {
    return DEFAULT_EXTERNAL_ID;
  }

  @Override
  public void serialize(@NotNull T stub, @NotNull StubOutputStream dataStream) throws IOException {
  }

  @Override
  public @NotNull T deserialize(@NotNull StubInputStream dataStream, StubElement parentStub) throws IOException {
    return (T)new PsiFileStubImpl(null);
  }

  @Override
  public void indexStub(@NotNull PsiFileStub stub, @NotNull IndexSink sink) {
  }

  public boolean shouldBuildStubFor(VirtualFile file) {
    return true;
  }

  public static int getTemplateStubBaseVersion() {
    return TEMPLATE_STUB_BASE_VERSION.getValue().intValue();
  }

  private static int calcTemplateStubBaseVersion() {
    IElementType[] dataElementTypes = IElementType.enumerate(
      (elementType) -> elementType instanceof IStubFileElementType && !(elementType.getLanguage() instanceof TemplateLanguage));
    return Arrays.stream(dataElementTypes).mapToInt((e) -> ((IStubFileElementType<?>)e).getStubVersion()).sum();
  }

  @ApiStatus.Internal
  public static void dropTemplateStubBaseVersion() {
    TEMPLATE_STUB_BASE_VERSION.drop();
  }
}