/*
 * Copyright 2010-2019 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.codegen.state

import ksp.com.intellij.openapi.project.Project
import ksp.org.jetbrains.kotlin.codegen.*
import ksp.org.jetbrains.kotlin.codegen.extensions.ClassFileFactoryFinalizerExtension
import ksp.org.jetbrains.kotlin.codegen.inline.GlobalInlineContext
import ksp.org.jetbrains.kotlin.codegen.inline.InlineCache
import ksp.org.jetbrains.kotlin.codegen.optimization.OptimizationClassBuilderFactory
import ksp.org.jetbrains.kotlin.codegen.serialization.JvmSerializationBindings
import ksp.org.jetbrains.kotlin.config.CompilerConfiguration
import ksp.org.jetbrains.kotlin.config.incrementalCompilationComponents
import ksp.org.jetbrains.kotlin.config.messageCollector
import ksp.org.jetbrains.kotlin.config.moduleName
import ksp.org.jetbrains.kotlin.descriptors.ClassDescriptor
import ksp.org.jetbrains.kotlin.descriptors.DeclarationDescriptor
import ksp.org.jetbrains.kotlin.descriptors.ModuleDescriptor
import ksp.org.jetbrains.kotlin.descriptors.VariableDescriptorWithAccessors
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporterFactory
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticSink
import ksp.org.jetbrains.kotlin.load.kotlin.incremental.components.IncrementalCache
import ksp.org.jetbrains.kotlin.modules.TargetId
import ksp.org.jetbrains.kotlin.psi.KtClassOrObject
import ksp.org.jetbrains.kotlin.psi.KtFile
import ksp.org.jetbrains.kotlin.resolve.BindingContext
import ksp.org.jetbrains.kotlin.resolve.BindingTrace
import ksp.org.jetbrains.kotlin.resolve.DelegatingBindingTrace
import ksp.org.jetbrains.kotlin.resolve.deprecation.DeprecationResolver
import ksp.org.jetbrains.kotlin.resolve.deprecation.DeprecationSettings
import ksp.org.jetbrains.kotlin.resolve.jvm.diagnostics.JvmDeclarationOrigin
import ksp.org.jetbrains.kotlin.storage.LockBasedStorageManager
import ksp.org.jetbrains.kotlin.types.KotlinType
import ksp.org.jetbrains.kotlin.types.TypeApproximator
import ksp.org.jetbrains.org.objectweb.asm.Type
import java.lang.reflect.InvocationTargetException

class GenerationState(
    val project: Project,
    val module: ModuleDescriptor,
    val configuration: CompilerConfiguration,
    builderFactory: ClassBuilderFactory = ClassBuilderFactories.BINARIES,
    val generateDeclaredClassFilter: GenerateClassFilter? = null,
    val targetId: TargetId? = null,
    moduleName: String? = configuration.moduleName,
    val jvmBackendClassResolver: JvmBackendClassResolver = JvmBackendClassResolverForModuleWithDependencies(module),
    val ignoreErrors: Boolean = false,
    diagnosticReporter: DiagnosticReporter? = null,
    compiledCodeProvider: CompiledCodeProvider = CompiledCodeProvider.Empty
) {
    val diagnosticReporter: DiagnosticReporter =
        diagnosticReporter ?: DiagnosticReporterFactory.createReporter(configuration.messageCollector)

    abstract class GenerateClassFilter {
        abstract fun shouldGenerateClass(processingClassOrObject: KtClassOrObject): Boolean
        abstract fun shouldGeneratePackagePart(ktFile: KtFile): Boolean
    }

    val config = JvmBackendConfig(configuration)

    val inlineCache: InlineCache = InlineCache(compiledCodeProvider)

    val incrementalCacheForThisTarget: IncrementalCache? = configuration.incrementalCompilationComponents?.let { components ->
        val targetId = targetId
            ?: moduleName?.let {
                // hack for Gradle IC, Gradle does not use build.xml file, so there is no way to pass target id
                TargetId(it, "java-production")
            } ?: error("Target ID should be specified for incremental compilation")
        components.getIncrementalCache(targetId)
    }

    val deprecationProvider = DeprecationResolver(
        LockBasedStorageManager.NO_LOCKS, config.languageVersionSettings, DeprecationSettings.Default
    )

    val moduleName: String = moduleName ?: JvmCodegenUtil.getModuleName(module)
    val classBuilderMode: ClassBuilderMode = builderFactory.classBuilderMode
    val bindingTrace: BindingTrace = DelegatingBindingTrace(BindingContext.EMPTY, "trace in GenerationState")
    val localDelegatedProperties: MutableMap<Type, List<VariableDescriptorWithAccessors>> = mutableMapOf()

    val globalInlineContext: GlobalInlineContext = GlobalInlineContext()
    val factory: ClassFileFactory = ClassFileFactory(
        this,
        BuilderFactoryForDuplicateClassNameDiagnostics(
            if (classBuilderMode.generateBodies) OptimizationClassBuilderFactory(builderFactory, this) else builderFactory,
            this
        ).let {
            loadClassBuilderInterceptors().fold(it) { classBuilderFactory: ClassBuilderFactory, extension ->
                extension.interceptClassBuilderFactory(classBuilderFactory, BindingContext.EMPTY, DiagnosticSink.DO_NOTHING)
            }
        },
        ClassFileFactoryFinalizerExtension.getInstances(project),
    )

    val globalSerializationBindings = JvmSerializationBindings()
    lateinit var mapInlineClass: (ClassDescriptor) -> Type

    class MultiFieldValueClassUnboxInfo(val unboxedTypesAndMethodNamesAndFieldNames: List<Triple<Type, String, String>>) {
        val unboxedTypes = unboxedTypesAndMethodNamesAndFieldNames.map { (type, _, _) -> type }
        val unboxedMethodNames = unboxedTypesAndMethodNamesAndFieldNames.map { (_, methodName, _) -> methodName }
    }

    var multiFieldValueClassUnboxInfo: (ClassDescriptor) -> MultiFieldValueClassUnboxInfo? = { null }

    lateinit var reportDuplicateClassNameError: (JvmDeclarationOrigin, String, String) -> Unit

    val typeApproximator: TypeApproximator = TypeApproximator(module.builtIns, config.languageVersionSettings)

    val newFragmentCaptureParameters: MutableList<Triple<String, KotlinType, DeclarationDescriptor>> = mutableListOf()

    @Suppress("UNCHECKED_CAST", "DEPRECATION_ERROR")
    private fun loadClassBuilderInterceptors(): List<org.jetbrains.kotlin.codegen.extensions.ClassBuilderInterceptorExtension> {
        val adapted = try {
            // Using Class.forName here because we're in the old JVM backend, and we need to load extensions declared in the JVM IR backend.
            Class.forName("org.jetbrains.kotlin.backend.jvm.extensions.ClassBuilderExtensionAdapter")
                .getDeclaredMethod("getExtensions", Project::class.java)
                .invoke(null, project) as List<org.jetbrains.kotlin.codegen.extensions.ClassBuilderInterceptorExtension>
        } catch (e: InvocationTargetException) {
            // Unwrap and rethrow any exception that happens. It's important e.g. in case of ProcessCanceledException.
            throw e.targetException
        }

        return org.jetbrains.kotlin.codegen.extensions.ClassBuilderInterceptorExtension.getInstances(project) + adapted
    }
}
