/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.backend.jvm

import ksp.org.jetbrains.kotlin.backend.common.actualizer.IrExtraActualDeclarationExtractor
import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.fir.backend.Fir2IrClassifierStorage
import ksp.org.jetbrains.kotlin.fir.backend.Fir2IrComponents
import ksp.org.jetbrains.kotlin.fir.declarations.FirDeclarationOrigin
import ksp.org.jetbrains.kotlin.fir.java.JavaSymbolProvider
import ksp.org.jetbrains.kotlin.fir.java.javaSymbolProvider
import ksp.org.jetbrains.kotlin.fir.languageVersionSettings
import ksp.org.jetbrains.kotlin.ir.declarations.IrClass
import ksp.org.jetbrains.kotlin.ir.declarations.IrDeclarationWithName
import ksp.org.jetbrains.kotlin.ir.declarations.IrPackageFragment
import ksp.org.jetbrains.kotlin.ir.symbols.IrClassSymbol
import ksp.org.jetbrains.kotlin.ir.symbols.IrSymbol
import ksp.org.jetbrains.kotlin.ir.util.classIdOrFail
import ksp.org.jetbrains.kotlin.name.CallableId

class FirDirectJavaActualDeclarationExtractor private constructor(
    private val javaSymbolProvider: JavaSymbolProvider,
    private val classifierStorage: Fir2IrClassifierStorage,
) : IrExtraActualDeclarationExtractor() {

    companion object {
        fun initializeIfNeeded(platformComponents: Fir2IrComponents): FirDirectJavaActualDeclarationExtractor? {
            val javaSymbolProvider = platformComponents.session.javaSymbolProvider
            if (javaSymbolProvider != null &&
                platformComponents.session.languageVersionSettings.supportsFeature(LanguageFeature.MultiPlatformProjects) &&
                platformComponents.session.languageVersionSettings.supportsFeature(LanguageFeature.DirectJavaActualization)
            ) {
                return FirDirectJavaActualDeclarationExtractor(javaSymbolProvider, platformComponents.classifierStorage)
            }
            return null
        }
    }

    override fun extract(expectIrClass: IrClass): IrClassSymbol? {
        val javaActualDeclaration = javaSymbolProvider.getClassLikeSymbolByClassId(expectIrClass.classIdOrFail)
            ?.takeIf { it.origin is FirDeclarationOrigin.Java.Source }
        if (javaActualDeclaration != null) {
            return classifierStorage.getIrClassSymbol(javaActualDeclaration)
        }
        return null
    }

    override fun extract(
        expectTopLevelCallables: List<IrDeclarationWithName>,
        expectCallableId: CallableId,
    ): List<IrSymbol> = emptyList()
}
