/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.expressions.impl

import ksp.org.jetbrains.kotlin.KtSourceElement
import ksp.org.jetbrains.kotlin.fir.declarations.FirAnonymousFunction
import ksp.org.jetbrains.kotlin.fir.expressions.FirAnnotation
import ksp.org.jetbrains.kotlin.fir.expressions.FirAnonymousFunctionExpression
import ksp.org.jetbrains.kotlin.fir.expressions.RawFirApi
import ksp.org.jetbrains.kotlin.fir.expressions.UnresolvedExpressionTypeAccess
import ksp.org.jetbrains.kotlin.fir.types.ConeKotlinType
import ksp.org.jetbrains.kotlin.fir.types.coneTypeOrNull
import ksp.org.jetbrains.kotlin.fir.visitors.FirTransformer
import ksp.org.jetbrains.kotlin.fir.visitors.FirVisitor
import ksp.org.jetbrains.kotlin.utils.addToStdlib.shouldNotBeCalled

@OptIn(UnresolvedExpressionTypeAccess::class)
internal class FirAnonymousFunctionExpressionImpl(
    override val source: KtSourceElement?,
    override var anonymousFunction: FirAnonymousFunction,
    override var isTrailingLambda: Boolean,
) : FirAnonymousFunctionExpression() {

    @UnresolvedExpressionTypeAccess
    override val coneTypeOrNull: ConeKotlinType?
        get() = anonymousFunction.typeRef.coneTypeOrNull

    override val annotations: List<FirAnnotation>
        get() = anonymousFunction.annotations

    override fun <R, D> acceptChildren(visitor: FirVisitor<R, D>, data: D) {
        anonymousFunction.accept(visitor, data)
    }

    override fun <D> transformChildren(transformer: FirTransformer<D>, data: D): FirAnonymousFunctionExpressionImpl {
        return transformAnonymousFunction(transformer, data)
    }

    override fun replaceAnnotations(newAnnotations: List<FirAnnotation>) {
        anonymousFunction.replaceAnnotations(newAnnotations)
    }

    @RawFirApi
    override fun replaceIsTrailingLambda(newIsTrailingLambda: Boolean) {
        isTrailingLambda = newIsTrailingLambda
    }

    override fun <D> transformAnnotations(transformer: FirTransformer<D>, data: D): FirAnonymousFunctionExpressionImpl {
        return this
    }

    override fun <D> transformAnonymousFunction(transformer: FirTransformer<D>, data: D): FirAnonymousFunctionExpressionImpl {
        anonymousFunction = anonymousFunction.transform(transformer, data)
        return this
    }

    override fun replaceAnonymousFunction(newAnonymousFunction: FirAnonymousFunction) {
        anonymousFunction = newAnonymousFunction
    }

    override fun replaceConeTypeOrNull(newConeTypeOrNull: ConeKotlinType?) {
        shouldNotBeCalled("anonymousFunction.replaceTypeRef() should be called instead")
    }
}
